#pragma once

#include <library/cpp/cgiparam/cgiparam.h>
#include <library/cpp/http/misc/httpcodes.h>
#include <library/cpp/http/misc/httpreqdata.h>
#include <library/cpp/logger/global/global.h>

#include <util/datetime/base.h>
#include <util/generic/buffer.h>
#include <util/generic/object_counter.h>
#include <util/memory/blob.h>

enum class EDeadlineCorrectionResult {
    dcrOK,
    dcrNoDeadline,
    dcrRequestExpired,
    dcrIncorrectDeadline,
};

class TServerRequestData;

class IReplyContext: public TObjectCounter<IReplyContext> {
private:
    static std::atomic<ui64> GlobalRequestsCounter;
    ui64 RequestId;
    TInstant RequestDeadline = TInstant::Max();
    TBlob OverrideBuf;

protected:
    virtual const TBlob& DoGetBuf() const = 0;

public:
    using TPtr = TAtomicSharedPtr<IReplyContext>;

public:
    IReplyContext() {
        RequestId = ++GlobalRequestsCounter;
    }
    virtual ~IReplyContext() = default;

    virtual const TCgiParameters& GetCgiParameters() const = 0;
    virtual TCgiParameters& MutableCgiParameters() = 0;

    TInstant GetRequestDeadline() const {
        Y_ASSERT(RequestDeadline != TInstant::Max());
        return RequestDeadline;
    }

    IReplyContext& SetRequestDeadline(TInstant deadline) {
        RequestDeadline = deadline;
        return *this;
    }

    virtual TInstant GetRequestStartTime() const = 0;

    virtual bool IsHttp() const = 0;
    virtual bool IsLocal() const { return false; }
    virtual IOutputStream& Output() = 0;

    ui64 GetRequestId() const {
        return RequestId;
    }

    void SetBuf(const TString& buf) {
        OverrideBuf = TBlob::FromString(buf);
    }

    const TBlob& GetBuf() const {
        return OverrideBuf.Empty() ? DoGetBuf() : OverrideBuf;
    }

    virtual const TServerRequestData& GetRequestData() const = 0;
    virtual TServerRequestData& MutableRequestData() = 0;

    virtual TStringBuf GetUri() const = 0;

    virtual void AddReplyInfo(const TString& key, const TString& value) = 0;
    virtual void MakeSimpleReply(const TBuffer& buf, int code = HTTP_OK) = 0;

    EDeadlineCorrectionResult DeadlineCorrection(const TDuration& scatterTimeout);
};
