#include "replier.h"
#include "exception.h"

#include <rtline/util/types/exception.h>

IHttpReplier::IHttpReplier(IReplyContext::TPtr context, const THttpStatusManagerConfig* config)
    : HttpStatusConfig(config ? *config : THttpStatusManagerConfig())
    , Context(context)
{
}

void IHttpReplier::Process(void* /*ThreadSpecificResource*/) {
    THolder<IHttpReplier> this_(this);
    SearchAndReply();
}

void IHttpReplier::Reply() {
    THolder<IHttpReplier> cleanup(this);
    IThreadPool* queue = DoSelectHandler();
    if (!queue) {
        SearchAndReply();
        return;
    }

    if (queue->Add(cleanup.Get())) {
        Y_UNUSED(cleanup.Release());
    } else {
        OnQueueFailure();
    }
}

void IHttpReplier::OnQueueFailure() {
    Y_ASSERT(Context);
    MakeErrorPage(HTTP_SERVICE_UNAVAILABLE, "Cannot queue request");
}

void IHttpReplier::MakeErrorPage(ui32 code, const TString& error) {
    ::MakeErrorPage(Context, code, error);
}

void IHttpReplier::SearchAndReply() {
    try {
        auto deadlineCheckResult = Context->DeadlineCorrection(GetDefaultTimeout());
        if (deadlineCheckResult == EDeadlineCorrectionResult::dcrRequestExpired) {
            TString additionalMessage = "no info";
            try {
                OnRequestExpired();
            } catch (const std::exception& e) {
                additionalMessage = FormatExc(e);
            }
            throw TCodedException(HttpStatusConfig.TimeoutStatus) << "request timeout: " << additionalMessage;
        } else if (deadlineCheckResult == EDeadlineCorrectionResult::dcrIncorrectDeadline) {
            throw TCodedException(HttpStatusConfig.SyntaxErrorStatus) << "incorrect &timeout=<value>";
        }
        DoSearchAndReply();
    } catch (const TCodedException& se) {
        MakeErrorPage(se.GetCode(), se.what());
    } catch (const std::exception& e) {
        MakeErrorPage(HttpStatusConfig.UnknownErrorStatus, e.what());
        WARNING_LOG << "Unexpected exception in SearchAndReply [" << Context->GetRequestId() << "]: " << e.what() << Endl;
    }
}
