#include "config.h"

#include <library/cpp/json/json_reader.h>
#include <library/cpp/logger/global/global.h>
#include <library/cpp/mediator/global_notifications/system_status.h>
#include <library/cpp/string_utils/url/url.h>
#include <library/cpp/testing/unittest/tests_data.h>
#include <util/stream/file.h>
#include <util/string/vector.h>

void TSelfHttpRequesterConfig::Init(const TYandexConfig::Section* section) {
    const auto& directives = section->GetDirectives();
    HeaderWhitelist = StringSplitter(
        directives.Value("HeaderWhitelist", JoinStrings(HeaderWhitelist.begin(), HeaderWhitelist.end(), ","))
    ).SplitBySet(" ,").SkipEmpty();
    Https = section->GetDirectives().Value<bool>("Https", Https);

    {
        TString envsStr;
        envsStr = section->GetDirectives().Value<TString>("Environments", envsStr);
        for (auto&& envStr : SplitString(envsStr, ",")) {
            EEnv env;
            CHECK_WITH_LOG(TryFromString(envStr, env));
            Environments.push_back(env);
        }
    }

    DelegatorAuthHeader = directives.Value<TString>("DelegatorAuthHeader", DelegatorAuthHeader);
    if (!DelegatorAuthHeader) {
        TString delegatorAuthHeaderPath;
        delegatorAuthHeaderPath = section->GetDirectives().Value<TString>("DelegatorAuthHeaderPath", delegatorAuthHeaderPath);
        if (!!delegatorAuthHeaderPath) {
            DelegatorAuthHeader = Strip(TFileInput(delegatorAuthHeaderPath).ReadAll());
        }
    }

    const TYandexConfig::TSectionsMap children = section->GetAllChildren();
    {
        auto it = children.find("RequestConfig");
        if (it != children.end()) {
            RequestConfig.InitFromSection(it->second);
        }
    }
    RequestTimeout = section->GetDirectives().Value<TDuration>("RequestTimeout", RequestTimeout);
    Https = section->GetDirectives().Value<bool>("Https", Https);
}

void TSelfHttpRequesterConfig::ToString(IOutputStream& os) const {
    os << "HeaderWhitelist: " << JoinStrings(HeaderWhitelist.begin(), HeaderWhitelist.end(), ",") << Endl;
    os << "Https: " << Https << Endl;
    {
        TString envsStr;
        for (auto&& env : Environments) {
            if (!envsStr) {
                envsStr += ",";
            }
            envsStr += ::ToString(env);
        }
        os << "Environments: " << envsStr << Endl;
    }

    os << "<RequestConfig>" << Endl;
    RequestConfig.ToString(os);
    os << "</RequestConfig>" << Endl;
    os << "RequestTimeout: " << RequestTimeout << Endl;
}

TSelfHttpRequesterConfig ParseFromString(const TString& configStr) {
    TSelfHttpRequesterConfig result;
    TAnyYandexConfig config;
    CHECK_WITH_LOG(config.ParseMemory(configStr.data()));
    result.Init(config.GetRootSection());
    return result;
}

TString TSelfHttpRequesterConfig::GetHost(const EEnv env) const {
    switch (env) {
        case EEnv::Admin:
            return "admin.carsharing.yandex.net";
        case EEnv::Prestable:
            return "prestable.carsharing.yandex.net";
        case EEnv::Stable:
            return "stable.carsharing.yandex.net";
        case EEnv::Testing:
            return "testing.carsharing.yandex.net";
        case EEnv::TestLocal:
            return "localhost";
        default:
            return "";
    }
}

ui32 TSelfHttpRequesterConfig::GetPort(const EEnv env) const {
    if (env != EEnv::TestLocal) {
        return Https ? 443 : 80;
    }

    ui32 serverPort;
    TFileInput fileInput("./test_port");
    fileInput >> serverPort;

    return serverPort;
}
