#include "model.h"

bool TSocialProfile::DeserializeFromJson(const NJson::TJsonValue& json) {
    if (json.Has("addresses")) {
        if (!json["addresses"].IsArray()) {
            return false;
        }
        for (auto&& address : json["addresses"].GetArray()) {
            if (!address.IsString()) {
                continue;
            }
            Addresses.push_back(address.GetString());
        }
    }
    JREAD_STRING(json, "provider", Provider);
    JREAD_STRING(json, "provider_code", ProviderCode);
    JREAD_STRING_NULLABLE_OPT(json, "username", Username);
    JREAD_UINT_OPT(json, "profile_id", ProfileId);
    JREAD_UINT_OPT(json, "uid", Uid);
    JREAD_STRING_OPT(json, "userid", Userid);
    return true;
}

NJson::TJsonValue TSocialProfile::SerializeToJson() const {
    NJson::TJsonValue result;
    JWRITE_DEF(result, "provider", Provider, "");
    JWRITE_DEF(result, "provider_code", ProviderCode, "");
    JWRITE_DEF(result, "uid", Uid, 0);
    JWRITE_DEF(result, "profile_id", ProfileId, 0);
    JWRITE_DEF(result, "userid", Userid, "");
    if (Username) {
        result["username"] = Username;
    } else {
        result["username"] = NJson::JSON_NULL;
    }
    NJson::TJsonValue addresses = NJson::JSON_ARRAY;
    for (auto&& address : Addresses) {
        addresses.AppendValue(address);
    }
    result["addresses"] = addresses;
    return result;
}

TSocialProfilesData TSocialProfilesData::FromHttpReply(const TString& content, const ui32 code) {
    NJson::TJsonValue json;
    if (code / 100 != 2 || !NJson::ReadJsonFastTree(content, &json) || !json.Has("profiles") || !json["profiles"].IsArray()) {
        return TSocialProfilesData({}, true);
    }

    bool broken = false;
    TVector<TSocialProfile> profiles;
    for (auto&& profileJson : json["profiles"].GetArray()) {
        TSocialProfile profile;
        if (!profile.DeserializeFromJson(profileJson)) {
            broken = true;
            continue;
        }
        profiles.emplace_back(std::move(profile));
    }

    return TSocialProfilesData(profiles, broken);
}

NJson::TJsonValue TSocialProfilesData::SerializeToJson() const {
    NJson::TJsonValue profilesList = NJson::JSON_ARRAY;
    for (auto&& profile : Profiles) {
        profilesList.AppendValue(profile.SerializeToJson());
    }
    NJson::TJsonValue result;
    result["profiles"] = std::move(profilesList);
    return result;
}
