#include "client.h"

#include <rtline/library/json/builder.h>

bool TSolomonClient::GetData(const TString& program, const TInstant from, const TInstant to, TVector<TTimeseries>& timeseries, TMessagesCollector& errors) const {
    NJson::TJsonValue reply;
    if (!SendRequest(ESolomonOperationType::GetData, CreateGetDataRequest(program, from, to), reply, errors)) {
        return false;
    }
    for (const auto& element : reply["vector"].GetArray()) {
        TTimeseries currectTimeseries;
        if (!currectTimeseries.FromJson(element)) {
            errors.AddMessage("parse", "cannot parse " + element.GetStringRobust());
            return false;
        }
        timeseries.emplace_back(std::move(currectTimeseries));
    }
    return true;
}

template<class TValue, class TValues = TVector<TValue>>
class TIterator {
    ui32 Id;
    TValues Elements;
    ui32 ElementId = 0;

public:
    TIterator(const ui32 vectorId, TValues&& values)
        : Id(vectorId)
        , Elements(std::move(values))
    {}

    ui32 GetId() const {
        return Id;
    }

    const TValue* Current() const {
        if (ElementId < Elements.size()) {
            return &Elements[ElementId];
        }
        return nullptr;
    }

    const TValue* Next() {
        if (++ElementId < Elements.size()) {
            return &Elements[ElementId];
        }
        return nullptr;
    }
};

bool TSolomonClient::GetData(const TString& program, const TInstant from, const TInstant to, const EAggregationType aggregationType, TTimeseries& timeseriesResult, TMessagesCollector& errors) const {
    TVector<TTimeseries> timeseriesVector;
    if (!GetData(program, from, to, timeseriesVector, errors)) {
        return false;
    }
    TVector<TIterator<TTimeseries::TPoint, TTimeseries>> iterators;
    std::multimap<TInstant, ui32> timeseriesProfile;
    for(size_t i = 0; i < timeseriesVector.size(); ++i) {
        TIterator<TTimeseries::TPoint, TTimeseries> iterator(i, std::move(timeseriesVector[i]));
        auto elementPtr = iterator.Current();
        if (!!elementPtr) {
            timeseriesProfile.emplace(elementPtr->GetTimestamp(), i);
        }
        iterators.emplace_back(std::move(iterator));
    }

    TVector<TTimeseries::TPoint> points;
    while (!timeseriesProfile.empty()) {
        auto& vectorIterator = iterators[timeseriesProfile.begin()->second];
        auto currentPointPtr = vectorIterator.Current();
        if (!currentPointPtr) {
            ERROR_LOG << "Internal error" << Endl;
            return false;
        }
        if (!points.empty() && points.back().GetTimestamp() == currentPointPtr->GetTimestamp()) {
            if (!TTimeseries::Aggregate(points.back(), *currentPointPtr, points.back(), aggregationType)) {
                ERROR_LOG << "Incorrect  aggregation" << Endl;
                return false;
            }
        } else {
            points.emplace_back(*currentPointPtr);
        }
        timeseriesProfile.erase(timeseriesProfile.begin());
        auto nextPointPtr = vectorIterator.Next();
        if (!!nextPointPtr) {
            timeseriesProfile.emplace(nextPointPtr->GetTimestamp(), vectorIterator.GetId());
        }
    }
    timeseriesResult.SetPoints(points);
    return true;
}

NNeh::THttpRequest TSolomonClient::CreateGetDataRequest(const TString& program, const TInstant from, const TInstant to) const {
    NJson::TJsonValue json = NJson::TMapBuilder("from", ::ToString(from))("to", ::ToString(to))("program", program);
    return TBase::CreateCommonRequest("/api/v2/projects/" + Config.GetProjectId() + "/sensors/data", ERequestMethod::POST, json, ERequestContentType::Json);
}
