#pragma once
#include <rtline/library/deprecated/async_impl/client.h>
#include <rtline/util/json_processing.h>

enum ESolomonOperationType {
    GetData /* "get_data" */,
};

class TSolomonClientConfig : public TTokenAuthRequestConfig {
    R_READONLY(TString, ProjectId);
public:
    void Init(const TYandexConfig::Section* section, const TMap<TString, NSimpleMeta::TConfig>* requestPolicy) {
        TTokenAuthRequestConfig::Init(section, requestPolicy);
        ProjectId = section->GetDirectives().Value("ProjectId", ProjectId);
        AssertCorrectConfig(!!ProjectId, "incorrect project id");
    }
    void ToString(IOutputStream& os) const {
        TTokenAuthRequestConfig::ToString(os);
        os << "ProjectId: " << ProjectId << Endl;
    }
};

class TSolomonClient : public TRequestClient<TSolomonClientConfig, TRequestLogger<ESolomonOperationType>, ESolomonOperationType> {
public:
    class TTimeseries {
    public:
        class TPoint {
        public:
            class TValue {
                R_READONLY(ui32, Value, 0);

                // deprecated
                double Count = 0;
                double Min = 0;
                double Max = 0;
                double Sum = 0;
                double Last = 0;

            public:
                double GetCount() const {
                    return Value ? Value : Count;
                }

                double GetMin() const {
                    return Value ? Value : Min;
                }

                double GetMax() const {
                    return Value ? Value : Max;
                }

                double GetSum() const {
                    return Value ? Value : Sum;
                }

                double GetLast() const {
                    return Value ? Value : Last;
                }

                bool FromJson(const NJson::TJsonValue& json) {
                    if (json.IsInteger()) {
                        Value = json.GetInteger();
                        return true;
                    }
                    JREAD_DOUBLE(json, "count", Count);
                    JREAD_DOUBLE(json, "max", Max);
                    JREAD_DOUBLE(json, "min", Min);
                    JREAD_DOUBLE(json, "sum", Sum);
                    JREAD_DOUBLE(json, "last", Last);
                    return true;
                }

                template<class TAction>
                void Aggregate(const TValue& a, const TValue& b, TAction action) {
                    Count = action(a.GetCount(), b.GetCount());
                    Min = action(a.GetMin(), b.GetMin());
                    Max = action(a.GetMax(), b.GetMax());
                    Sum = action(a.GetSum(), b.GetSum());
                    Last = action(a.GetLast(), b.GetLast());
                }
            };

        public:
            R_FIELD(TInstant, Timestamp);
            R_FIELD(TValue, Value);

        public:
            bool Build(const TInstant timestamp, const NJson::TJsonValue& json) {
                Timestamp = timestamp;
                return Value.FromJson(json);
            }

            template<class TAction>
            bool Aggregate(const TPoint& a, const TPoint& b, TAction action) {
                if (a.GetTimestamp() != b.GetTimestamp()) {
                    return false;
                }
                Timestamp = a.GetTimestamp();
                Value.Aggregate(a.GetValue(), b.GetValue(), action);
                return true;
            }
        };

    public:
        R_FIELD(TVector<TPoint>, Points);

    public:
        size_t size() const {
            return Points.size();
        }

        const TPoint& operator[](ui32 i) const {
            return Points[i];
        }

        bool FromJson(const NJson::TJsonValue& json) {
            auto& timestamps = json["timeseries"]["timestamps"].GetArray();
            auto& values = json["timeseries"]["values"].GetArray();
            if (timestamps.size() != values.size()) {
                return false;
            }
            for (size_t i = 0; i < timestamps.size(); ++i) {
                if (!timestamps[i].IsUInteger()) {
                    return false;
                }
                if (!values[i].IsMap() && !values[i].IsInteger()) {
                    continue;
                }
                TPoint point;
                if (!point.Build(TInstant::MilliSeconds(timestamps[i].GetUInteger()), values[i])) {
                    return false;
                }
                Points.emplace_back(std::move(point));
            }
            return true;
        }

        static bool Aggregate(const TPoint& a, const TPoint& b, TPoint& result, const EAggregationType aggregationType) {
            switch (aggregationType) {
            case EAggregationType::Average:
                return result.Aggregate(a, b, [](double a, double b) {
                    return (a + b) / 2.;
                });
            case EAggregationType::Max:
                return result.Aggregate(a, b, [](double a, double b) {
                    return Max<double>(a, b);
                });
            case EAggregationType::Min:
                return result.Aggregate(a, b, [](double a, double b) {
                    return Min<double>(a, b);
                });
            case EAggregationType::Sum:
            case EAggregationType::SumOne:
                return result.Aggregate(a, b, [](double a, double b) {
                    return a + b;
                });
            case EAggregationType::LastValue:
                result = a;
                return true;
            default:
                ERROR_LOG << "Incorrect aggregation type" << Endl;
                return false;
            }
            return true;
        }
    };
    using TBase = TRequestClient<TSolomonClientConfig, TRequestLogger<ESolomonOperationType>, ESolomonOperationType>;

public:
    TSolomonClient(const TSolomonClientConfig& config)
        : TBase(config, "solomon_api")
    {}

    bool GetData(const TString& program, const TInstant from, const TInstant to, TVector<TTimeseries>& timeseries, TMessagesCollector& errors) const;
    bool GetData(const TString& program, const TInstant from, const TInstant to, const EAggregationType aggregationType, TTimeseries& timeseries, TMessagesCollector& errors) const;

private:
    NNeh::THttpRequest CreateGetDataRequest(const TString& program, const TInstant from, const TInstant to) const;
};
