#include "client.h"

#include <aws/core/auth/AWSCredentials.h>

#include <aws/sqs/model/CreateQueueRequest.h>
#include <aws/sqs/model/CreateQueueResult.h>
#include <aws/sqs/model/SendMessageRequest.h>
#include <aws/sqs/model/SendMessageResult.h>
#include <aws/sqs/model/ReceiveMessageRequest.h>
#include <aws/sqs/model/ReceiveMessageResult.h>
#include <aws/sqs/model/DeleteMessageRequest.h>

TSQSClient::TSQSClient(const TSQSClientConfig& config)
    : Config(config)
    , Logger("sqs-api")
    , AwsGuard(InitializeAws())
{
}

bool TSQSClient::SendMessage(const TString& queue, const TString& body, TMessagesCollector& errors) const {
    Logger.ProcessStart(ESQSOperationType::SendMessage);

    auto smOut = SQSClient->SendMessage(CreateSendMessageRequest(queue, body));

    if (!smOut.IsSuccess()) {
        Logger.ProcessError(ESQSOperationType::SendMessage, "Error sending message: " + TString(smOut.GetError().GetMessage()), errors);
        return false;
    }

    return true;
}

bool TSQSClient::RecieveMessages(const TString& queue, TVector<TString>& messages, TVector<Aws::String>& receiptHandles, TMessagesCollector& errors) const {
    Logger.ProcessStart(ESQSOperationType::ReceiveMessages);

    auto rmOut = SQSClient->ReceiveMessage(CreateReceiveMessagesRequest(queue));

    if (!rmOut.IsSuccess()) {
        Logger.ProcessError(ESQSOperationType::ReceiveMessages, "Error receiving messages: " + TString(rmOut.GetError().GetMessage()), errors);
        return false;
    }

    const auto& rawMessages = rmOut.GetResult().GetMessages();
    for (const auto& rawMessage : rawMessages) {
        messages.push_back(TString(rawMessage.GetBody()));
        receiptHandles.push_back(rawMessage.GetReceiptHandle());
    }

    return true;
}

bool TSQSClient::RecieveMessages(const TString& queue, TVector<TString>& messages, TMessagesCollector& errors, bool removeOriginal) const {
    TVector<Aws::String> receiptHandles;
    if (!RecieveMessages(queue, messages, receiptHandles, errors)) {
        return false;
    }
    return (!removeOriginal) || DeleteMessages(queue, receiptHandles, errors);
}

bool TSQSClient::DeleteMessages(const TString& queue, const TVector<Aws::String>& receiptHandles, TMessagesCollector& errors) const {
    for (const auto& receiptHandle : receiptHandles) {
        Logger.ProcessStart(ESQSOperationType::DeleteMessage);

        auto dmOut = SQSClient->DeleteMessage(CreateDeleteMessageRequest(queue, receiptHandle));
        if (!dmOut.IsSuccess()) {
            Logger.ProcessError(ESQSOperationType::DeleteMessage, "Error deleting message " + TString(receiptHandle) + ": " + TString(dmOut.GetError().GetMessage()), errors);
            return false;
        }
    }
    return true;
}

TAtomicSharedPtr<Aws::SQS::SQSClient> TSQSClient::CreateSQSClientInstance(const TSQSClientConfig& config) const {
    Aws::Client::ClientConfiguration sqsClientConfiguration;
    sqsClientConfiguration.endpointOverride = config.GetHost() + ":" + ToString(config.GetPort());
    sqsClientConfiguration.scheme = (config.GetIsHttps()) ? Aws::Http::Scheme::HTTPS : Aws::Http::Scheme::HTTP;

    Aws::Auth::AWSCredentials credentials;
    credentials.SetAWSAccessKeyId(config.GetAccount());
    credentials.SetAWSSecretKey("unused");
    credentials.SetSessionToken(config.GetToken());

    return MakeAtomicShared<Aws::SQS::SQSClient>(credentials, sqsClientConfiguration);
}

Aws::SQS::Model::SendMessageRequest TSQSClient::CreateSendMessageRequest(const TString& queue, const TString& body) const {
    Aws::SQS::Model::SendMessageRequest smRequest;
    smRequest.SetQueueUrl(Config.CreateQueueUrl(queue));
    smRequest.SetMessageBody(body);
    return smRequest;
}

Aws::SQS::Model::ReceiveMessageRequest TSQSClient::CreateReceiveMessagesRequest(const TString& queue) const {
    Aws::SQS::Model::ReceiveMessageRequest rmRequest;
    rmRequest.SetQueueUrl(Config.CreateQueueUrl(queue));
    rmRequest.SetMaxNumberOfMessages(Config.GetMaxNumberOfMessages());
    rmRequest.SetWaitTimeSeconds(Config.GetRequestTimeout().Seconds());
    return rmRequest;
}

Aws::SQS::Model::DeleteMessageRequest TSQSClient::CreateDeleteMessageRequest(const TString& queue, const Aws::String& receiptHandle) const {
    Aws::SQS::Model::DeleteMessageRequest dmRequest;
    dmRequest.SetQueueUrl(Config.CreateQueueUrl(queue));
    dmRequest.SetReceiptHandle(receiptHandle);
    return dmRequest;
}
