#pragma once

#include <rtline/util/json_processing.h>
#include <rtline/util/types/accessor.h>

#include <library/cpp/json/json_reader.h>

#include <util/generic/map.h>
#include <util/generic/serialized_enum.h>
#include <util/generic/vector.h>
#include <util/string/builder.h>
#include <util/string/cast.h>
#include <util/string/join.h>
#include <util/string/split.h>
#include <util/string/strip.h>

class TStaffEntry {
public:
    enum class EStaffEntryField {
        Uid /* "uid" */,
        Username /* "login" */,
        Name /* "name" */,
        WorkPhone /* "work_phone" */,
        WorkEmail /* "work_email" */,
        MobilePhone /* "phones" */,
        IsDeleted /* "is_deleted" */,
        IsDismissed /* "official.is_dismissed" */,
        QuitAt /* "official.quit_at" */,
        DepartmentUrl /* "department_group.url" */
    };

    using TStaffEntryFields = TSet<EStaffEntryField>;

    enum class EStaffNameLocale {
        EN /* "en" */,
        RU /* "ru" */
    };

    class TStaffName {
        R_READONLY(TString, FirstName);
        R_READONLY(TString, LastName);
        R_READONLY(TString, MiddleName);

    public:
        bool DeserializeFromJson(const NJson::TJsonValue& data, EStaffNameLocale locale) {
            FirstName = ReadNameFieldSafe(data, "first", locale);
            LastName = ReadNameFieldSafe(data, "last", locale);
            MiddleName = ReadNameFieldSafe(data, "middle", locale);
            return true;
        }

    private:
        TString ReadNameFieldSafe(const NJson::TJsonValue& data, const TString& name, EStaffNameLocale locale) const {
            TString result = "";
            if (data.IsMap() && data.Has(name)) {
                auto localizedNames = data[name].GetMap();
                auto needle = localizedNames.FindPtr(::ToString(locale));
                if (needle && needle->IsString()) {
                    result = Strip(needle->GetString());
                }
            }
            return result;
        }
    };

    using TStaffNames = TMap<EStaffNameLocale, TStaffName>;

    R_READONLY(ui64, Uid, 0);
    R_READONLY(TString, Username);
    R_READONLY(TStaffNames, Name);
    R_READONLY(ui64, WorkPhone, 0);
    R_READONLY(TString, WorkEmail);
    R_READONLY(TString, MainMobilePhone);
    R_READONLY(TVector<TString>, MobilePhones);
    R_READONLY(bool, Deleted, false);
    R_READONLY(bool, Dismissed, false);
    R_READONLY(TInstant, QuitAt, TInstant::Zero());
    R_READONLY(TString, DepartmentUrl);

public:
    TStaffEntry(const TStaffEntryFields& fields = {})
        : Fields(fields)
    {
    }

    bool DeserializeFromJson(const NJson::TJsonValue& data);

    bool HasField(EStaffEntryField field) const {
        return !Fields || Fields.contains(field);
    }

    TString GetFieldsStr() const {
        if (!!Fields) {
            return JoinSeq(",", Fields);
        }

        TVector<TString> fields;
        for (const auto& [_, name] : GetEnumNames<EStaffEntryField>()) {
            fields.push_back(name);
        }
        return JoinSeq(",", fields);
    }

private:
    inline bool GetNestedField(const NJson::TJsonValue& data, const TString& name, const TString& subname, TString& result) const;
    inline bool GetNestedField(const NJson::TJsonValue& data, const TString& name, const TString& subname, bool& result) const;
    inline bool GetNestedField(const NJson::TJsonValue& data, const TString& name, const TString& subname, NJson::TJsonValue& result) const;

    const TStaffEntryFields Fields;
};


class TStaffEntrySelector {
public:
    enum class EStaffEntryField {
        Uid /* "uid" */,
        Username /* "login" */,
        WorkPhone /* "work_phone" */,
        WorkEmail /* "work_email" */,
        DepartmentUrl /* "department_url" */
    };

    R_READONLY(EStaffEntryField, Type);
    R_READONLY(TString, Value);

public:
    TStaffEntrySelector(EStaffEntryField type, const TString& value)
        : Type(type)
        , Value(value)
    {
    }

    template <typename TContainer>
    TStaffEntrySelector(EStaffEntryField type, const TContainer& values)
        : TStaffEntrySelector(type, JoinSeq(",", values))
    {
    }

    TString ToCgiParameterString() const;
};
