#include <drive/library/cpp/staff/client.h>
#include <drive/library/cpp/staff/config.h>

#include <library/cpp/testing/unittest/tests_data.h>
#include <library/cpp/testing/unittest/registar.h>

#include <util/string/builder.h>
#include <util/system/env.h>
#include <util/generic/vector.h>

namespace {
    TStaffClientConfig GetStaffConfig() {
        auto config = TStaffClientConfig::ParseFromString(
            TStringBuilder()
            << "Host: staff-api.test.yandex-team.ru" << Endl
            << "Port: 443" << Endl
            << "IsHttps: true" << Endl
            << "Account: robot-carsharing" << Endl
            << "Token: " << GetEnv("TESTING_STAFF_TOKEN") << Endl
            << "TokenPath: " << GetEnv("TESTING_STAFF_TOKEN_PATH") << Endl
            << "RequestTimeout: 5s" << Endl
            << "<RequestConfig>" << Endl
            << "    MaxAttempts: 1" << Endl
            << "</RequestConfig>" << Endl
        );
        return config;
    }

    TStaffClient GetStaffClient() {
        return TStaffClient(GetStaffConfig());
    }
}

Y_UNIT_TEST_SUITE(StaffClient) {
    Y_UNIT_TEST(GetUserDataByLogin) {
        auto client = GetStaffClient();

        TVector<TStaffEntry> results;
        TMessagesCollector errors;

        const TString username = "robot-carsharing";

        UNIT_ASSERT_C(
            client.GetUserData(
                TStaffEntrySelector(TStaffEntrySelector::EStaffEntryField::Username, username),
                results,
                errors,
                {} /* all fields from enum */
            ),
            "error getting staff data: " << errors.GetStringReport()
        );
        UNIT_ASSERT_EQUAL(results.size(), 1);

        auto userData = results.front();

        UNIT_ASSERT_STRINGS_EQUAL(userData.GetUsername(), username);
        UNIT_ASSERT_STRINGS_EQUAL(userData.GetWorkEmail(), "robot-carsharing@yandex-team.ru");
        UNIT_ASSERT(!userData.GetWorkPhone());
        UNIT_ASSERT(!userData.GetMainMobilePhone());
        UNIT_ASSERT_EQUAL(userData.GetMobilePhones().size(), 0);
        UNIT_ASSERT_STRINGS_EQUAL(userData.GetDepartmentUrl(), "virtual_robots");
        UNIT_ASSERT_VALUES_EQUAL(userData.GetUid(), 1120000000067712ul);
        UNIT_ASSERT_VALUES_EQUAL(userData.GetQuitAt(), TInstant::Zero());
        UNIT_ASSERT(!userData.IsDeleted());
        UNIT_ASSERT(!userData.IsDismissed());

        auto userNames = userData.GetName();

        UNIT_ASSERT(userNames.contains(TStaffEntry::EStaffNameLocale::EN));
        UNIT_ASSERT_VALUES_EQUAL(userNames[TStaffEntry::EStaffNameLocale::EN].GetFirstName(), username);
        UNIT_ASSERT_VALUES_EQUAL(userNames[TStaffEntry::EStaffNameLocale::EN].GetLastName(), username);
        UNIT_ASSERT_VALUES_EQUAL(userNames[TStaffEntry::EStaffNameLocale::EN].GetMiddleName(), "");

        UNIT_ASSERT(userNames.contains(TStaffEntry::EStaffNameLocale::RU));
        UNIT_ASSERT_VALUES_EQUAL(userNames[TStaffEntry::EStaffNameLocale::RU].GetFirstName(), username);
        UNIT_ASSERT_VALUES_EQUAL(userNames[TStaffEntry::EStaffNameLocale::RU].GetLastName(), username);
        UNIT_ASSERT_VALUES_EQUAL(userNames[TStaffEntry::EStaffNameLocale::RU].GetMiddleName(), "");
    }

    Y_UNIT_TEST(GetUserDataByUid) {
        auto client = GetStaffClient();

        TVector<TStaffEntry> results;
        TMessagesCollector errors;

        const TString username = "robot-carsharing";
        const ui64 uid = 1120000000067712ul;

        UNIT_ASSERT_C(
            client.GetUserData(
                TStaffEntrySelector(TStaffEntrySelector::EStaffEntryField::Uid, ToString(uid)),
                results,
                errors,
                {} /* all fields from enum */
            ),
            "error getting staff data: " << errors.GetStringReport()
        );
        UNIT_ASSERT_EQUAL(results.size(), 1);

        auto userData = results.front();

        UNIT_ASSERT_STRINGS_EQUAL(userData.GetUsername(), username);
        UNIT_ASSERT_VALUES_EQUAL(userData.GetUid(), uid);
    }

    Y_UNIT_TEST(GetUserDataByWorkEmail) {
        auto client = GetStaffClient();

        TVector<TStaffEntry> results;
        TMessagesCollector errors;

        const TString username = "robot-carsharing";
        const TString workEmail = "robot-carsharing@yandex-team.ru";

        UNIT_ASSERT_C(
            client.GetUserData(
                TStaffEntrySelector(TStaffEntrySelector::EStaffEntryField::WorkEmail, workEmail),
                results,
                errors,
                {} /* all fields from enum */
            ),
            "error getting staff data: " << errors.GetStringReport()
        );
        UNIT_ASSERT_EQUAL(results.size(), 1);

        auto userData = results.front();

        UNIT_ASSERT_STRINGS_EQUAL(userData.GetUsername(), username);
        UNIT_ASSERT_VALUES_EQUAL(userData.GetWorkEmail(), workEmail);
    }

    Y_UNIT_TEST(GetUserDataByWorkPhone) {
        auto client = GetStaffClient();

        TVector<TStaffEntry> results;
        TMessagesCollector errors;

        const TString username = "mg";
        const ui64 workPhone = 4105;

        UNIT_ASSERT_C(
            client.GetUserData(
                TStaffEntrySelector(TStaffEntrySelector::EStaffEntryField::WorkPhone, ToString(workPhone)),
                results,
                errors,
                {} /* all fields from enum */
            ),
            "error getting staff data: " << errors.GetStringReport()
        );
        UNIT_ASSERT_EQUAL(results.size(), 1);

        auto userData = results.front();

        UNIT_ASSERT_STRINGS_EQUAL(userData.GetUsername(), username);
        UNIT_ASSERT_VALUES_EQUAL(userData.GetWorkPhone(), workPhone);
    }

    Y_UNIT_TEST(GetUserDataFromDepatment) {
        auto client = GetStaffClient();

        TVector<TStaffEntry> results;
        TMessagesCollector errors;

        const TString departmentUrl = "yandex_content_8006";  // Yandex Drive
        const size_t limit = 50;

        UNIT_ASSERT_C(
            client.GetUserData(
                TStaffEntrySelector(TStaffEntrySelector::EStaffEntryField::DepartmentUrl, departmentUrl),
                results,
                errors,
                {} /* all fields from enum */,
                "uid",
                limit /* to split on pages */
            ),
            "error getting staff data: " << errors.GetStringReport()
        );
        UNIT_ASSERT_GE(results.size(), limit);  // total number of Yandex Drive employees
    }
}
