#include <drive/library/cpp/startrek/client.h>

#include <library/cpp/json/json_reader.h>
#include <rtline/util/json_processing.h>

#include <util/string/builder.h>
#include <util/string/subst.h>

void TStartrekRequestCallback::ProcessResponseData() {
    if (!IsRequestSuccessful(Code)) {
        if (Result.Has("errorMessages") && Result["errorMessages"].IsArray()) {
            if (ErrorMessage) {
                ErrorMessage += "; ";
            }
            ErrorMessage += Result["errorMessages"].GetStringRobust();
        }
        if (Result.Has("errors") && Result["errors"].IsMap()) {
            if (ErrorMessage) {
                ErrorMessage += "; ";
            }
            ErrorMessage += Result["errors"].GetStringRobust();
        }
        if (!ErrorMessage) {
            ErrorMessage = "Unknown error";
        }
    }
}

TString TStartrekClient::GetTicketUri(const TString& ticketKey, const bool useApiHost) const {
    if (!ticketKey) {
        return "";
    }
    return TStringBuilder()
           << ((Config.GetIsHttps()) ? "https://" : "http://")
           << ((useApiHost) ? Config.GetHost() : SubstGlobalCopy(Config.GetHost(), "st-api.", "st."))
           << "/" << ticketKey;
}

TString TStartrekClient::GetTicketUri(const TTicket& ticket, const bool useApiHost) const {
    TString ticketKey;
    if (!ticket.GetAdditionalValue(::ToString(TTicket::ETicketField::Key), ticketKey)) {
        return "";
    }
    return GetTicketUri(ticketKey, useApiHost);
}

TString TStartrekClient::GetAttachmentUploadUri() const {
    return ((Config.GetIsHttps()) ? "https://" : "http://") + Config.GetHost() + "/v2/attachments";
}

bool TStartrekClient::GetIssueInfo(const TString& issueName, TTicket& result, TMessagesCollector& errors) const {
    NJson::TJsonValue rawResult;
    if (!SendRequest(EStartrekOperationType::GetIssue, CreateGetIssueRequest(issueName), rawResult, errors)) {
        return false;
    }
    return result.DeserializeFromJson(rawResult);
}

bool TStartrekClient::CreateIssue(const TTicket& content, TTicket& result, TMessagesCollector& errors) const {
    NJson::TJsonValue rawResult;
    if (!SendRequest(EStartrekOperationType::CreateIssue, CreateMakeIssueRequest(content.SerializeToJson()), rawResult, errors)) {
        return false;
    }
    return result.DeserializeFromJson(rawResult);
}

bool TStartrekClient::PatchIssue(const TString& issueName, const TTicket& update, TTicket& result, TMessagesCollector& errors) const {
    NJson::TJsonValue rawResult;
    if (!SendRequest(EStartrekOperationType::PatchIssue, CreatePatchIssueRequest(issueName, update.SerializeToJson()), rawResult, errors)) {
        return false;
    }
    return result.DeserializeFromJson(rawResult);
}

bool TStartrekClient::SearchIssue(const TString& query, TTickets& tickets, TMessagesCollector& errors) const {
    NJson::TJsonValue rawResult;
    if (!SendRequest(EStartrekOperationType::SearchIssue, CreateSearchIssueRequest(query), rawResult, errors)) {
        return false;
    }
    if (!rawResult.IsArray()) {
        return false;
    }
    for (const auto& rawTicket : rawResult.GetArray()) {
        TTicket ticket;
        if (!ticket.DeserializeFromJson(rawTicket)) {
            return false;
        }
        tickets.push_back(std::move(ticket));
    }
    return true;
}

bool TStartrekClient::GetIssueAttachments(const TString& issueName, TAttachments& attachments, TMessagesCollector& errors) const {
    NJson::TJsonValue rawResult;
    if (!SendRequest(EStartrekOperationType::GetIssueAttachments, CreateGetIssueAttachmentsRequest(issueName), rawResult, errors)) {
        return false;
    }
    for (const auto& item : rawResult.GetArray()) {
        TAttachment attachment;
        if (!attachment.DeserializeFromJson(item)) {
            return false;
        }
        attachments.push_back(std::move(attachment));
    }
    return true;
}

bool TStartrekClient::UploadAttachment(const TString& fileData, TAttachmentId& attachmentId, TMessagesCollector& errors) const {
    return UploadAttachment("", fileData, attachmentId, errors);
}

bool TStartrekClient::UploadAttachment(const TString& fileName, const TString& fileData, TAttachmentId& attachmentId, TMessagesCollector& errors) const {
    NJson::TJsonValue rawResult;
    if (!SendRequest(EStartrekOperationType::UploadAttachment, CreateUploadAttachmentRequest(fileName, fileData), rawResult, errors)) {
        return false;
    }
    return TJsonProcessor::Read(rawResult, "id", attachmentId, /* mustBe = */ true);
}

bool TStartrekClient::AddComment(const TString& issueName, const TComment& comment, TMessagesCollector& errors) const {
    TComment result;
    return AddComment(issueName, comment, result, errors);
}

bool TStartrekClient::AddComment(const TString& issueName, const TComment& comment, TComment& result, TMessagesCollector& errors) const {
    NJson::TJsonValue rawResult;  // comment meta info or error info
    if (!SendRequest(EStartrekOperationType::AddComment, CreateAddCommentRequest(issueName, comment.SerializeToJson()), rawResult, errors)) {
        return false;
    }
    return result.DeserializeFromJson(rawResult);
}

bool TStartrekClient::GetComment(const TString& issueName, const TString& commentId, TComment& result, TMessagesCollector& errors) const {
    NJson::TJsonValue rawResult;
    if (!SendRequest(EStartrekOperationType::GetComment, CreateGetCommentRequest(issueName, commentId), rawResult, errors)) {
        return false;
    }
    return result.DeserializeFromJson(rawResult);
}

bool TStartrekClient::GetAllComments(const TString& issueName, TComments& result, TMessagesCollector& errors) const {
    NJson::TJsonValue rawResult;
    if (!SendRequest(EStartrekOperationType::GetAllComments, CreateGetAllCommentsRequest(issueName), rawResult, errors)) {
        return false;
    }
    for (const auto& item : rawResult.GetArray()) {
        TComment comment;
        if (!comment.DeserializeFromJson(item)) {
            return false;
        }
        result.push_back(std::move(comment));
    }
    return true;
}

bool TStartrekClient::DeleteComment(const TString& issueName, const TString& commentId, TMessagesCollector& errors) const {
    NJson::TJsonValue rawResult;
    return SendRequest(EStartrekOperationType::DeleteComment, CreateDeleteCommentRequest(issueName, commentId), rawResult, errors);
}

bool TStartrekClient::GetTransitions(const TString& issueName, TTransitions& result, TMessagesCollector& errors) const {
    NJson::TJsonValue transitions;
    if (!SendRequest(EStartrekOperationType::GetTransitions, CreateGetTransitionsRequest(issueName), transitions, errors)) {
        return false;
    }

    if (!transitions.IsArray()) {
        errors.AddMessage(__LOCATION__, "transitions have unexpected format");
        return false;
    }

    for (const auto& rawTransition : transitions.GetArray()) {
        TTransition transition;
        if (!transition.DeserializeFromJson(rawTransition)) {
            return false;
        }
        result.push_back(std::move(transition));
    }

    return true;
}

bool TStartrekClient::ExecuteTransition(const TString& issueName, const TString& transition, TMessagesCollector& errors, const TTicket& update) const {
    NJson::TJsonValue result;  // updated transitions or error info
    return SendRequest(EStartrekOperationType::ExecuteTransition, CreateExecuteTransitionRequest(issueName, transition, update.SerializeToJson()), result, errors);
}

NNeh::THttpRequest TStartrekClient::CreateGetIssueRequest(const TString& issueName) const {
    return CreateCommonRequest("v2/issues/" + issueName);
}

NNeh::THttpRequest TStartrekClient::CreateMakeIssueRequest(const NJson::TJsonValue& content) const {
    return CreateCommonRequest("v2/issues/", ERequestMethod::POST, content, ERequestContentType::Json);
}

NNeh::THttpRequest TStartrekClient::CreatePatchIssueRequest(const TString& issueName, const NJson::TJsonValue& update) const {
    return CreateCommonRequest("v2/issues/" + issueName, ERequestMethod::PATCH, update, ERequestContentType::Json);
}

NNeh::THttpRequest TStartrekClient::CreateSearchIssueRequest(const TString& query) const {
    NJson::TJsonMap data({{ "query", query }});
    return CreateCommonRequest("v2/issues/_search", ERequestMethod::POST, data, ERequestContentType::Json);
}

NNeh::THttpRequest TStartrekClient::CreateGetIssueAttachmentsRequest(const TString& issueName) const {
    return CreateCommonRequest("v2/issues/" + issueName + "/attachments");
}

NNeh::THttpRequest TStartrekClient::CreateUploadAttachmentRequest(const TString& fileName, const TString& fileData) const {
    auto request = CreateCommonRequest("v2/attachments");
    if (fileName) {
        request.SetCgiData("filename=" + fileName);
    }
    request.SetFile(fileData, fileName);
    return request;
}

NNeh::THttpRequest TStartrekClient::CreateAddCommentRequest(const TString& issueName, const NJson::TJsonValue& commentData) const {
    return CreateCommonRequest("v2/issues/" + issueName + "/comments", ERequestMethod::POST, commentData, ERequestContentType::Json);
}

NNeh::THttpRequest TStartrekClient::CreateGetCommentRequest(const TString& issueName, const TString& commentId) const {
    return CreateCommonRequest("v2/issues/" + issueName + "/comments/" + commentId);
}

NNeh::THttpRequest TStartrekClient::CreateGetAllCommentsRequest(const TString& issueName) const {
    return CreateCommonRequest("v2/issues/" + issueName + "/comments");
}

NNeh::THttpRequest TStartrekClient::CreateDeleteCommentRequest(const TString& issueName, const TString& commentId) const {
    return CreateCommonRequest("v2/issues/" + issueName + "/comments/" + commentId, ERequestMethod::DELETE, {}, ERequestContentType::Json);
}

NNeh::THttpRequest TStartrekClient::CreateGetTransitionsRequest(const TString& issueName) const {
    return CreateCommonRequest("v2/issues/" + issueName + "/transitions");
}

NNeh::THttpRequest TStartrekClient::CreateExecuteTransitionRequest(const TString& issueName, const TString& transition, const NJson::TJsonValue& data) const {
    return CreateCommonRequest("v2/issues/" + issueName + "/transitions/" + transition + "/_execute", ERequestMethod::POST, data, ERequestContentType::Json);
}
