#pragma once

#include "config.h"
#include "entity.h"
#include "logger.h"

#include <library/cpp/json/json_reader.h>
#include <library/cpp/string_utils/url/url.h>

#include <rtline/library/deprecated/async_impl/client.h>

#include <util/generic/cast.h>
#include <util/string/cast.h>

class TStartrekRequestCallback : public TRequestCallback<EStartrekOperationType, TStartrekLogger> {
    using TBase = TRequestCallback<EStartrekOperationType, TStartrekLogger>;

public:
    TStartrekRequestCallback(NJson::TJsonValue& result, ui32& code, TString& errorMessage, const EStartrekOperationType type, const TStartrekLogger& logger)
        : TBase(result, code, errorMessage, type, logger)
    {
    }

    virtual void ProcessResponseData() override;
};

class TStartrekClient : public TRequestClient<TStartrekClientConfig, TStartrekLogger, EStartrekOperationType, TStartrekRequestCallback> {
    using TBase = TRequestClient<TStartrekClientConfig, TStartrekLogger, EStartrekOperationType, TStartrekRequestCallback>;

public:
    using TTicket = TStartrekTicket;
    using TTickets = TVector<TStartrekTicket>;
    using TComment = TStartrekComment;
    using TComments = TVector<TStartrekComment>;
    using TTransition = TStartrekTransition;
    using TTransitions = TVector<TTransition>;
    using TAttachment = TStartrekAttachment;
    using TAttachmentId = TAttachment::TId;
    using TAttachments = TVector<TAttachment>;

    TStartrekClient(const TStartrekClientConfig& config)
        : TBase(config, "startrek_api")
    {
    }

    TString GetTicketUri(const TString& ticketKey, const bool useApiHost = false) const;
    TString GetTicketUri(const TTicket& ticket, const bool useApiHost = false) const;
    TString GetAttachmentUploadUri() const;

    bool GetIssueInfo(const TString& issueName, TTicket& result, TMessagesCollector& errors) const;
    bool CreateIssue(const TTicket& content,  TTicket& result, TMessagesCollector& errors) const;
    bool PatchIssue(const TString& issueName, const TTicket& update, TTicket& result, TMessagesCollector& errors) const;

    bool SearchIssue(const TString& query, TTickets& tickets, TMessagesCollector& errors) const;

    bool GetIssueAttachments(const TString& issueName, TAttachments& attachments, TMessagesCollector& errors) const;

    bool UploadAttachment(const TString& fileData, TAttachmentId& attachmentId, TMessagesCollector& errors) const;
    bool UploadAttachment(const TString& fileName, const TString& fileData, TAttachmentId& attachmentId, TMessagesCollector& errors) const;

    bool AddComment(const TString& issueName, const TComment& comment, TMessagesCollector& errors) const;
    bool AddComment(const TString& issueName, const TComment& comment, TComment& result, TMessagesCollector& errors) const;
    bool GetComment(const TString& issueName, const TString& commentId, TComment& result, TMessagesCollector& errors) const;
    bool GetAllComments(const TString& issueName, TComments& result, TMessagesCollector& errors) const;
    bool DeleteComment(const TString& issueName, const TString& commentId, TMessagesCollector& errors) const;

    bool GetTransitions(const TString& issueName, TTransitions& result, TMessagesCollector& errors) const;
    bool ExecuteTransition(const TString& issueName, const TString& transition, TMessagesCollector& errors, const TTicket& update = Default<TTicket>()) const;

private:
    NNeh::THttpRequest CreateGetIssueRequest(const TString& issueName) const;
    NNeh::THttpRequest CreateMakeIssueRequest(const NJson::TJsonValue& content) const;
    NNeh::THttpRequest CreatePatchIssueRequest(const TString& issueName, const NJson::TJsonValue& update) const;
    NNeh::THttpRequest CreateSearchIssueRequest(const TString& query) const;
    NNeh::THttpRequest CreateGetIssueAttachmentsRequest(const TString& issueName) const;
    NNeh::THttpRequest CreateUploadAttachmentRequest(const TString& fileName, const TString& fileData) const;
    NNeh::THttpRequest CreateAddCommentRequest(const TString& issueName, const NJson::TJsonValue& commentData) const;
    NNeh::THttpRequest CreateGetCommentRequest(const TString& issueName, const TString& commentId) const;
    NNeh::THttpRequest CreateGetAllCommentsRequest(const TString& issueName) const;
    NNeh::THttpRequest CreateDeleteCommentRequest(const TString& issueName, const TString& commentId) const;
    NNeh::THttpRequest CreateGetTransitionsRequest(const TString& issueName) const;
    NNeh::THttpRequest CreateExecuteTransitionRequest(const TString& issueName, const TString& transition, const NJson::TJsonValue& data = {}) const;
};
