#include <drive/library/cpp/startrek/entity.h>

#include <rtline/library/json/cast.h>
#include <rtline/library/json/parse.h>
#include <rtline/util/json_processing.h>

#include <library/cpp/json/json_reader.h>
#include <util/string/cast.h>

namespace {
    static NJson::TJsonValue FilterNonEmptyStrings(const NJson::TJsonValue& arrayValue) {
        NJson::TJsonArray filteredItems;

        for (const auto& innerValue : arrayValue.GetArray()) {
            if (innerValue.IsString() && !!innerValue.GetString()) {
                filteredItems.AppendValue(innerValue);
            }
        }

        return filteredItems;
    }
}

namespace NJson {
    template <>
    TJsonValue ToJson(const TStartrekTicket& object) {
        return object.SerializeToJson();
    }

    template <>
    bool TryFromJson(const TJsonValue& value, TStartrekTicket& result) {
        return result.DeserializeFromJson(value);
    }
}

const NJson::TJsonValue* TStartrekTicket::GetAdditionalValuePtr(const TString& path) const {
    const NJson::TJsonValue* valuePtr = nullptr;

    TStringBuf pathBuf(path);
    auto head = pathBuf.NextTok('.');

    if (AdditionalAttributes.contains(head)) {
        const NJson::TJsonValue& tree = AdditionalAttributes.at(head);

        if (!path.empty()) {
            valuePtr = tree.GetValueByPath(pathBuf);
        } else {
            valuePtr = &tree;
        }
    }

    return valuePtr;
}

TString TStartrekTicket::GetKey() const {
    return GetAdditionalValue<TString>(::ToString(ETicketField::Key)).GetOrElse("");
}

TString TStartrekTicket::GetQueue() const {
    return GetAdditionalValue<TString>(::ToString(ETicketField::Queue)).GetOrElse("");
}

bool TStartrekTicket::SetAdditionalValue(const TString& key, const NJson::TJsonValue& value) {
    AdditionalAttributes[key] = value;
    return true;
}

bool TStartrekTicket::DeserializeFromJson(const NJson::TJsonValue& data) {
    JREAD_STRING_NULLABLE_OPT(data, "description", Description);
    JREAD_STRING_NULLABLE_OPT(data, "summary", Summary);
    for (const auto& [attrName, attrValue] : data.GetMap()) {
        if (attrName != "description" && attrName != "summary") {
            SetAdditionalValue(attrName, attrValue);
        }
    }
    return true;
}

NJson::TJsonValue TStartrekTicket::SerializeToJson() const {
    NJson::TJsonValue result(NJson::JSON_MAP);
    if (!!Description) {
        JWRITE(result, "description", Description);
    }
    if (!!Summary) {
        JWRITE(result, "summary", Summary);
    }
    for (const auto& [attrName, attrValue] : AdditionalAttributes) {
        NJson::TJsonValue serializableValue;
        if (GetSerializableAdditionalAttribute(attrName, attrValue, serializableValue)) {
            JWRITE(result, attrName, std::move(serializableValue));
        }
    }
    return result;
}

bool TStartrekTicket::GetSerializableAdditionalAttribute(const TString& key, const NJson::TJsonValue& value, NJson::TJsonValue& serializableValue) const {
    if (!value.IsDefined()) {
        return false;
    }

    bool isSuccess = true;

    // filter empty tags and components
    if (key == ::ToString(EContainerTicketField::Tags) || key == ::ToString(EContainerTicketField::Components)) {
        if (value.IsArray()) {
            serializableValue = FilterNonEmptyStrings(value);
            isSuccess = !serializableValue.GetArray().empty();
        } else if (value.IsMap()) {
            serializableValue = NJson::TJsonMap();
            for (auto&& [action, itemsToProcess] : value.GetMap()) {  // action - add/set/remove
                NJson::TJsonValue serializableItemsToProcess = FilterNonEmptyStrings(itemsToProcess);
                if (!serializableItemsToProcess.GetArray().empty()) {
                    serializableValue.InsertValue(action, std::move(serializableItemsToProcess));
                }
            }
            isSuccess = !serializableValue.GetMap().empty();
        } else {
            isSuccess = false;  // actually type is erroneous and it won't be processed correctly
        }
    } else {
        serializableValue = value;
    }

    return isSuccess;
}

bool TStartrekComment::DeserializeFromJson(const NJson::TJsonValue& info) {
    JREAD_UINT_OPT(info, "id", Id);
    JREAD_INSTANT_ISOFORMAT_NULLABLE_OPT(info, "createdAt", CreatedAt);
    JREAD_INSTANT_ISOFORMAT_NULLABLE_OPT(info, "updatedAt", UpdatedAt);
    JREAD_STRING(info, "text", Text);
    for (const auto& item : info["summonees"].GetArray()) {
        if (item["id"].IsString()) {
            Summonees.push_back(item["id"].GetString());
        } else if (item.IsString()) {
            Summonees.push_back(item.GetString());
        }
    }
    for (const auto& item : info["attachments"].GetArray()) {
        if (item["id"].IsString()) {
            AttachmentIds.push_back(item["id"].GetString());
        } else if (item.IsString()) {
            AttachmentIds.push_back(item.GetString());
        }
    }
    return true;
}

bool TStartrekComment::DeserializeFromString(const TString& s) {
    // treat body either as comment text or serialized comment data
    NJson::TJsonValue commentData;
    if (!(NJson::ReadJsonTree(s, &commentData, /* throwOnError = */ false) && DeserializeFromJson(commentData))) {
        SetText(s);
    }
    return true;
}

NJson::TJsonValue TStartrekComment::SerializeToJson() const {
    NJson::TJsonValue commentInfo;
    if (!!Id) {
        JWRITE(commentInfo, "id", Id);
    }
    JWRITE_INSTANT_ISOFORMAT_DEF(commentInfo, "createdAt", CreatedAt, TInstant::Zero());
    JWRITE_INSTANT_ISOFORMAT_DEF(commentInfo, "updatedAt", UpdatedAt, TInstant::Zero());
    JWRITE(commentInfo, "text", Text);
    TJsonProcessor::WriteContainerArray(commentInfo, "summonees", Summonees, /* writeEmpty = */ false);
    TJsonProcessor::WriteContainerArray(commentInfo, "attachmentIds", AttachmentIds, /* writeEmpty = */ false);
    return commentInfo;
}

bool TStartrekTransition::DeserializeFromJson(const NJson::TJsonValue& info) {
    JREAD_STRING(info, "id", Name);
    JREAD_STRING(info, "display", DisplayName);
    if (!info.Has("to") || !info["to"].IsMap()) {
        return false;
    }
    JREAD_STRING(info["to"], "display", TargetStatusName);
    JREAD_STRING(info["to"], "key", TargetStatusKey);
    return true;
}

bool TStartrekAttachment::DeserializeFromJson(const NJson::TJsonValue& info) {
    return NJson::ParseField(info["id"], Id, true) &&
           NJson::ParseField(info["name"], Name, true) &&
           NJson::ParseField(info["content"], Url, true) &&
           NJson::ParseField(info["createdAt"], CreatedAt, true) &&
           NJson::ParseField(info["mimetype"], MimeType) &&
           NJson::ParseField(info["size"], Size);
}

NJson::TJsonValue TStartrekLink::SerializeToJson() const {
    NJson::TJsonValue result;
    NJson::InsertField(result, "issue", Issue);
    NJson::InsertField(result, "relationship", NJson::Stringify(RelationshipType));
    return result;
}
