#pragma once

#include "config.h"
#include "logger.h"

#include <library/cpp/json/json_reader.h>
#include <library/cpp/string_utils/url/url.h>

#include <rtline/library/deprecated/async_impl/client.h>
#include <rtline/library/json/cast.h>

#include <util/generic/cast.h>

class TStartrekTicket {
    using TAdditionalAttributes = THashMap<TString, NJson::TJsonValue>;

    R_FIELD(TString, Description);
    R_FIELD(TString, Summary);

protected:
    TAdditionalAttributes AdditionalAttributes;

public:
    // Field and transition descriptions are located here: https://st.yandex-team.ru/admin/fields
    enum class ETicketField {
        Key /* "key" */,
        Queue /* "queue" */,
        QueueKey /* "queue.key" */,
        StatusKey /* "status.key" */,
    };

    enum class EAdditionalValuesAction {
        SET /* "set" */,
        ADD /* "add" */,
        REMOVE /* "remove" */
    };

    enum class EContainerTicketField {
        Tags /* "tags" */,
        Components /* "components" */,
        Followers /* "followers" */,
        AttachmentIds /* "attachmentIds" */,
        Links /* "links" */,
    };

    TStartrekTicket() = default;
    explicit TStartrekTicket(const NJson::TJsonValue& data) {
        DeserializeFromJson(data);
    }
    virtual ~TStartrekTicket() = default;

    bool Empty() const {
        return !Description && !Summary && !AdditionalAttributes;
    }

    explicit operator bool() const {
        return !Empty();
    }

    virtual bool DeserializeFromJson(const NJson::TJsonValue& data);
    virtual NJson::TJsonValue SerializeToJson() const;

    template <typename T>
    static TMaybe<T> BuildFromJson(const NJson::TJsonValue& data) {
        T ticket;
        if (!ticket.DeserializeFromJson(data)) {
            return Nothing();
        }
        return ticket;
    }

    template <class T>
    const T* GetAs() const {
        return dynamic_cast<const T*>(this);
    }

    template <class T>
    const T& GetAsSafe() const {
        return *VerifyDynamicCast<const T*>(this);
    }

    TString GetKey() const;
    TString GetQueue() const;

    template <typename T>
    bool GetAdditionalValue(const TString& path, T& value) const;
    template <typename T>
    TMaybe<T> GetAdditionalValue(const TString& path) const;

    bool SetAdditionalValue(const TString& key, const NJson::TJsonValue& value);

    template <typename TContainer>
    void SetAdditionalContainerValue(const TString& key, const TContainer& values, bool modifyExisting = false);
    template <typename TContainer>
    void SetAdditionalContainerValue(EContainerTicketField key, const TContainer& values, bool modifyExisting = false);

    template <typename TContainer>
    void AddAdditionalContainerValue(const TString& key, const TContainer& values, bool modifyExisting = false);
    template <typename TContainer>
    void AddAdditionalContainerValue(EContainerTicketField key, const TContainer& values, bool modifyExisting = false);

    template <typename TContainer>
    void RemoveAdditionalContainerValue(const TString& key, const TContainer& values, bool modifyExisting = false);
    template <typename TContainer>
    void RemoveAdditionalContainerValue(EContainerTicketField key, const TContainer& values, bool modifyExisting = false);

protected:
    const NJson::TJsonValue* GetAdditionalValuePtr(const TString& path) const;

    template <typename TContainer>
    void OperateAdditionalContainerValue(const TString& key, EAdditionalValuesAction action, const TContainer& values, bool modifyExisting = false);

    bool GetSerializableAdditionalAttribute(const TString& key, const NJson::TJsonValue& value, NJson::TJsonValue& serializableValue) const;
};

template <typename T>
bool TStartrekTicket::GetAdditionalValue(const TString& path, T& value) const {
    const NJson::TJsonValue* valuePtr = GetAdditionalValuePtr(path);
    return (valuePtr) ? NJson::TryFromJson(*valuePtr, value) : false;
}

template <typename T>
TMaybe<T> TStartrekTicket::GetAdditionalValue(const TString& path) const {
    T value;
    if (GetAdditionalValue(path, value)) {
        return value;
    }
    return {};
}

template <typename TContainer>
void TStartrekTicket::OperateAdditionalContainerValue(const TString& key, EAdditionalValuesAction action, const TContainer& values, bool modifyExisting) {
    if (!AdditionalAttributes.contains(key)) {
        AdditionalAttributes[key] = NJson::JSON_ARRAY;
    }

    NJson::TJsonValue& actualValues = AdditionalAttributes[key];

    if (!modifyExisting) {
        TString actionName = ToString(action);
        if (!actualValues.Has(actionName)) {
            actualValues[actionName] = NJson::JSON_ARRAY;
        }
        for (const auto& value : values) {
            actualValues[actionName].AppendValue(value);
        }
    } else {
        switch (action) {
        case EAdditionalValuesAction::SET:
            actualValues = NJson::JSON_ARRAY;
            for (const auto& value : values) {
                actualValues.AppendValue(value);
            }
            break;
        case EAdditionalValuesAction::ADD:
            for (const auto& value : values) {
                actualValues.AppendValue(value);
            }
            break;
        case EAdditionalValuesAction::REMOVE:
            using TValue = typename TContainer::value_type;
            NJson::TJsonValue filtered(NJson::JSON_ARRAY);
            TSet<TValue> removeFilter(values.cbegin(), values.cend());
            for (const auto& i : actualValues.GetArray()) {
                TValue actualValue;
                if (TryFromString<TValue>(i.GetStringRobust(), actualValue) && removeFilter.contains(actualValue)) {
                    continue;
                }
                filtered.AppendValue(i);  // remains original value
            }
            AdditionalAttributes[key] = filtered;
            break;
        }
    }
}

template <typename TContainer>
void TStartrekTicket::SetAdditionalContainerValue(const TString& key, const TContainer& values, bool modifyExisting) {
    OperateAdditionalContainerValue<TContainer>(key, EAdditionalValuesAction::SET, values, modifyExisting);
}

template <typename TContainer>
void TStartrekTicket::SetAdditionalContainerValue(EContainerTicketField key, const TContainer& values, bool modifyExisting) {
    SetAdditionalContainerValue<TContainer>(::ToString(key), values, modifyExisting);
}

template <typename TContainer>
void TStartrekTicket::AddAdditionalContainerValue(const TString& key, const TContainer& values, bool modifyExisting) {
    OperateAdditionalContainerValue<TContainer>(key, EAdditionalValuesAction::ADD, values, modifyExisting);
}

template <typename TContainer>
void TStartrekTicket::AddAdditionalContainerValue(EContainerTicketField key, const TContainer& values, bool modifyExisting) {
    AddAdditionalContainerValue<TContainer>(::ToString(key), values, modifyExisting);
}

template <typename TContainer>
void TStartrekTicket::RemoveAdditionalContainerValue(const TString& key, const TContainer& values, bool modifyExisting) {
    OperateAdditionalContainerValue<TContainer>(key, EAdditionalValuesAction::REMOVE, values, modifyExisting);
}

template <typename TContainer>
void TStartrekTicket::RemoveAdditionalContainerValue(EContainerTicketField key, const TContainer& values, bool modifyExisting) {
    RemoveAdditionalContainerValue<TContainer>(::ToString(key), values, modifyExisting);
}

class TStartrekComment {
    using TSummonees = TVector<TString>;
    using TAttachmentIds = TVector<TString>;

    R_FIELD(ui64, Id, 0);
    R_FIELD(TInstant, CreatedAt, TInstant::Zero());
    R_FIELD(TInstant, UpdatedAt, TInstant::Zero());
    R_FIELD(TString, Text);
    R_FIELD(TSummonees, Summonees);
    R_FIELD(TAttachmentIds, AttachmentIds);

public:
    TStartrekComment() = default;
    TStartrekComment(const TString& text, const TSummonees& summonees = {}, const TAttachmentIds& attachmentIds = {})  // implicit conversion from TString
        : Text(text)
        , Summonees(summonees)
        , AttachmentIds(attachmentIds)
    {
    }

    explicit operator bool() const {
        return !!Text || !!AttachmentIds;
    }

    bool DeserializeFromJson(const NJson::TJsonValue& info);
    bool DeserializeFromString(const TString& s);
    NJson::TJsonValue SerializeToJson() const;
};

class TStartrekTransition {
    R_FIELD(TString, Name);
    R_FIELD(TString, DisplayName);
    R_FIELD(TString, TargetStatusKey);
    R_FIELD(TString, TargetStatusName);

public:
    bool DeserializeFromJson(const NJson::TJsonValue& info);
};

class TStartrekAttachment {
public:
    using TId = TString;

    bool DeserializeFromJson(const NJson::TJsonValue& info);

private:
    R_FIELD(TId, Id);
    R_FIELD(TString, Name);
    R_FIELD(TString, Url);
    R_FIELD(TInstant, CreatedAt);
    R_FIELD(TString, MimeType);
    R_FIELD(ui64, Size, 0);
};

class TStartrekLink {
public:
    enum class ERelationshipType {
        Relates /* "relates" */,
        IsDependentBy /* "is dependent by" */,
        DependsOn /* "depends on" */,
        IsSubtaskFor /* "is subtask for" */,
        IsParentTaskFor /* "is parent task for" */,
        Duplicates /* "duplicates" */,
        IsDuplicatedBy /* "is duplicated by" */,
        IsEpicOf /* "is epic of" */,
        HasEpic /* "has epic" */,
        Original /* "original" */,
        Clone /* "clone" */,
    };

    explicit TStartrekLink(const TString& issue, const ERelationshipType relationshipType = ERelationshipType::Relates)
        : Issue(issue)
        , RelationshipType(relationshipType)
    {
    }

    NJson::TJsonValue SerializeToJson() const;

private:
    R_FIELD(TString, Issue);
    R_FIELD(ERelationshipType, RelationshipType);
};
