#include <drive/library/cpp/startrek/client.h>
#include <drive/library/cpp/startrek/config.h>
#include <library/cpp/testing/unittest/tests_data.h>
#include <library/cpp/testing/unittest/registar.h>
#include <util/string/builder.h>
#include <util/system/env.h>
#include <util/generic/vector.h>

namespace {
    TStartrekClientConfig GetStartrekConfig() {
        auto config = TStartrekClientConfig::ParseFromString(
            TStringBuilder()
            << "Host: st-api.test.yandex-team.ru" << Endl
            << "Port: 443" << Endl
            << "IsHttps: true" << Endl
            << "Account: robot-carsharing" << Endl
            << "Token: " << GetEnv("TESTING_STARTREK_TOKEN") << Endl
            << "TokenPath: " << GetEnv("TESTING_STARTREK_TOKEN_PATH") << Endl
            << "RequestTimeout: 1s" << Endl
            << "<RequestConfig>" << Endl
            << "    MaxAttempts: 1" << Endl
            << "</RequestConfig>" << Endl
        );
        return config;
    }

    TStartrekClient GetStartrekClient() {
        return TStartrekClient(GetStartrekConfig());
    }

    TString GetTicketStatus(const TStartrekClient& client, const TString& issueName) {
        TStartrekTicket ticketInfo;
        TMessagesCollector obtainInfoErrors;
        UNIT_ASSERT_C(client.GetIssueInfo(issueName, ticketInfo, obtainInfoErrors), "error obtaining ticket info: " << obtainInfoErrors.GetStringReport());
        TString status;
        UNIT_ASSERT(ticketInfo.GetAdditionalValue(::ToString(TStartrekTicket::ETicketField::StatusKey), status));
        return status;
    }

    void CheckUpdateTicketTags(const TStartrekClient& client, const TString& issueName, const TStartrekTicket& ticketUpdate, const TVector<TString>& expectedTags) {
        TStartrekTicket updatedInfo;
        TMessagesCollector errors;
        UNIT_ASSERT(client.PatchIssue(issueName, ticketUpdate, updatedInfo, errors));

        TVector<TString> tags;
        updatedInfo.GetAdditionalValue(::ToString(TStartrekTicket::EContainerTicketField::Tags), tags);
        UNIT_ASSERT_VALUES_EQUAL(tags, expectedTags);
    }

    void CleanupOldComments(const TStartrekClient& client, const TString& issueName, const TDuration& limit = TDuration::Hours(1)) {
        TInstant now = TInstant::Now();

        TVector<ui64> commentIdsToDelete;
        do {
            commentIdsToDelete.clear();

            TStartrekClient::TComments comments;
            TMessagesCollector errors;
            UNIT_ASSERT_C(client.GetAllComments(issueName, comments, errors), "unable to get all comments: " << errors.GetStringReport());

            for (const auto& comment : comments) {
                if (now - comment.GetUpdatedAt() > limit) {
                    commentIdsToDelete.push_back(comment.GetId());
                }
            }

            for (const ui64 commentId : commentIdsToDelete) {
                client.DeleteComment(issueName, ToString(commentId), errors);
            }
        } while (!commentIdsToDelete.empty());
    }
}

Y_UNIT_TEST_SUITE(StartrekClient) {
    Y_UNIT_TEST(ProcessComment) {
        static const TString issueName = "DRIVETEST-1";

        auto client = GetStartrekClient();
        TString comment = "Startrek client test comment at "+ TInstant::Now().ToString();

        TStartrekComment result;
        TMessagesCollector errors;
        UNIT_ASSERT_C(client.AddComment(issueName, comment, result, errors), "error adding comment: " << errors.GetStringReport());

        TStartrekComment check;
        UNIT_ASSERT_C(client.GetComment(issueName, ToString(result.GetId()), check, errors), "error obtaining comment: " << errors.GetStringReport());
        UNIT_ASSERT_VALUES_EQUAL(result.GetId(), check.GetId());
        UNIT_ASSERT_STRINGS_EQUAL(comment, check.GetText());

        client.DeleteComment(issueName, ToString(result.GetId()), errors);
        UNIT_ASSERT_C(!client.GetComment(issueName, ToString(result.GetId()), check, errors), "comment has not been deleted");

        CleanupOldComments(client, issueName);
    }

    Y_UNIT_TEST(UpdateBody) {
        static const TString issueName = "DRIVETEST-1";
        static const size_t maxDescriptionLength = 4096;

        auto client = GetStartrekClient();

        TStartrekTicket ticketInfo;
        TMessagesCollector obtainInfoErrors;
        UNIT_ASSERT_C(client.GetIssueInfo(issueName, ticketInfo, obtainInfoErrors), "error obtaining ticket info: " << obtainInfoErrors.GetStringReport());

        TString description = ticketInfo.GetDescription();

        if (description.length() >= maxDescriptionLength) {
            size_t defaultOffset = description.length() - maxDescriptionLength;
            size_t offset = description.find('\n', defaultOffset);
            if (offset == TString::npos) {
                offset = defaultOffset;
            }
            description = description.substr(offset + 1);
        }

        description += "\nStartrek client update at "+ TInstant::Now().ToString();

        TStartrekTicket issueUpdate;
        issueUpdate.SetDescription(description);

        TStartrekTicket patchResult;
        TMessagesCollector updateErrors;
        UNIT_ASSERT_C(client.PatchIssue(issueName, issueUpdate, patchResult, updateErrors), "error updating ticket: " << updateErrors.GetStringReport());
        UNIT_ASSERT_STRINGS_EQUAL_C(description, patchResult.GetDescription(), "description set differs from target one");
    }

    Y_UNIT_TEST(ExecuteTransition) {
        static const TString issueName = "DRIVETEST-1";

        auto client = GetStartrekClient();
        TString status = GetTicketStatus(client, issueName);

        TString transition, targetStatus;
        if (status == "open") {
            transition = "start_progress";
            targetStatus = "inProgress";
        } else if (status == "inProgress") {
            transition = "stop_progress";
            targetStatus = "open";
        } else {
            UNIT_FAIL("cannot change ticket " << issueName << " status: status " << status << " is unknown");
        }

        TMessagesCollector executeTransitionErrors;
        UNIT_ASSERT_C(client.ExecuteTransition(issueName, transition, executeTransitionErrors), "error executing transition: " << executeTransitionErrors.GetStringReport());

        TString updatedStatus = GetTicketStatus(client, issueName);
        UNIT_ASSERT_STRINGS_EQUAL_C(updatedStatus, targetStatus, "status set differs from target one");
    }

    Y_UNIT_TEST(ExecuteInvalidTransition) {
        static const TString issueName = "DRIVETEST-1";

        auto client = GetStartrekClient();
        TString status = GetTicketStatus(client, issueName);

        TString transition;
        if (status == "open") {
            transition = "stop_progress";
        } else if (status == "inProgress") {
            transition = "start_progress";
        } else {
            UNIT_FAIL("cannot change ticket " << issueName << " status: status " << status << " is unknown");
        }

        TMessagesCollector executeTransitionErrors;
        UNIT_ASSERT(!client.ExecuteTransition(issueName, transition, executeTransitionErrors));
    }

    Y_UNIT_TEST(ProcessTicketTags) {
        static const TString issueName = "DRIVETEST-4";

        auto client = GetStartrekClient();

        TStartrekTicket ticketSetTagsUpdate;
        ticketSetTagsUpdate.SetAdditionalContainerValue(::ToString(TStartrekTicket::EContainerTicketField::Tags), TVector<TString>({"some_tag_1"}));
        CheckUpdateTicketTags(client, issueName, ticketSetTagsUpdate, {"some_tag_1"});

        TStartrekTicket ticketAddTagsUpdate;
        ticketAddTagsUpdate.AddAdditionalContainerValue(::ToString(TStartrekTicket::EContainerTicketField::Tags), TVector<TString>({"some_tag_2"}));
        CheckUpdateTicketTags(client, issueName, ticketAddTagsUpdate, {"some_tag_1", "some_tag_2"});

        TStartrekTicket ticketRemoveTagsUpdate;
        ticketRemoveTagsUpdate.RemoveAdditionalContainerValue(::ToString(TStartrekTicket::EContainerTicketField::Tags), TVector<TString>({"some_tag_1", "some_tag_2"}));
        CheckUpdateTicketTags(client, issueName, ticketRemoveTagsUpdate, {});
    }

    Y_UNIT_TEST(ProcessTicketTagsInplace) {
        static const TString issueName = "DRIVETEST-4";

        auto client = GetStartrekClient();
        TMessagesCollector errors;

        TStartrekTicket ticketInfo;
        UNIT_ASSERT(client.GetIssueInfo(issueName, ticketInfo, errors));

        ticketInfo.SetAdditionalContainerValue(::ToString(TStartrekTicket::EContainerTicketField::Tags), TVector<TString>({"some_tag_1"}), true);

        TVector<TString> setTags;
        ticketInfo.GetAdditionalValue(::ToString(TStartrekTicket::EContainerTicketField::Tags), setTags);
        UNIT_ASSERT_VALUES_EQUAL(setTags, TVector<TString>({"some_tag_1"}));

        ticketInfo.AddAdditionalContainerValue(::ToString(TStartrekTicket::EContainerTicketField::Tags), TVector<TString>({"some_tag_2"}), true);

        TVector<TString> addedTags;
        ticketInfo.GetAdditionalValue(::ToString(TStartrekTicket::EContainerTicketField::Tags), addedTags);
        UNIT_ASSERT_VALUES_EQUAL(addedTags, TVector<TString>({"some_tag_1", "some_tag_2"}));

        ticketInfo.RemoveAdditionalContainerValue(::ToString(TStartrekTicket::EContainerTicketField::Tags), TVector<TString>({"some_tag_1", "some_tag_2"}), true);

        TVector<TString> removedTags;
        ticketInfo.GetAdditionalValue(::ToString(TStartrekTicket::EContainerTicketField::Tags), removedTags);
        UNIT_ASSERT_VALUES_EQUAL(removedTags, TVector<TString>());
    }

    Y_UNIT_TEST(SearchIssue) {
        const TString query = "c02f4cf5-deb013c";

        auto client = GetStartrekClient();
        TMessagesCollector errors;

        TVector<TStartrekTicket> tickets;
        UNIT_ASSERT(client.SearchIssue(query, tickets, errors));
        UNIT_ASSERT(!tickets.empty());

        const TString needleKey = "DRIVETEST-2";

        bool found = false;
        for (auto&& ticket : tickets) {
            if (needleKey == ticket.GetKey()) {
                found = true;
                break;
            }
        }
        UNIT_ASSERT(found);
    }
}
