#include "client.h"

#include <rtline/library/json/builder.h>
#include <rtline/library/json/parse.h>
#include <rtline/library/unistat/cache.h>
#include <rtline/util/types/messages_collector.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/xml/document/xml-document.h>

#include <util/stream/str.h>
#include <util/string/builder.h>
#include <util/string/subst.h>


namespace {
    template <class T>
    class IApplier {
    public:
        using TResult = T;

    private:
        TString Method;

    public:
        IApplier(const TString& method)
            : Method(method)
        {
        }

        TResult operator()(const NThreading::TFuture<NUtil::THttpReply>& r) const {
            const auto& report = r.GetValue();
            TUnistatSignalsCache::SignalAdd(Method + "-suport-ai-codes", ToString(report.Code()), 1);
            return DoParseResult(report.EnsureSuccessfulReply());
        }

    private:
        virtual TResult DoParseResult(const NNeh::THttpReply& report) const = 0;
    };

    class TMakeCallApplier : public IApplier<TSupportAICall> {
        using TBase = IApplier<TSupportAICall>;

    private:
        TString Phone;

    public:
        TMakeCallApplier(const TString& phone)
            : TBase("call-init")
            , Phone(phone)
        {
        }

    private:
        virtual TBase::TResult DoParseResult(const NNeh::THttpReply& report) const override {
            NJson::TJsonValue rawResult;
            if (!NJson::ReadJsonFastTree(report.Content(), &rawResult)) {
                ythrow yexception() << "Error getting objects by phone(" << Phone << "): " << report.GetDebugReply();
            }
            TSupportAICall call;
            if (!call.DeserializeFromJson(rawResult)) {
                ythrow yexception() << "Fail to parse reply with phone(" << Phone << "): " << rawResult.GetStringRobust();
            }
            return call;
        }
    };

    class TCallResultApplier : public IApplier<NJson::TJsonValue> {
        using TBase = IApplier<NJson::TJsonValue>;

    private:
        TString CallId;

    public:
        TCallResultApplier(const TString& callId)
            : TBase("call-status")
            , CallId(callId)
        {
        }

    private:
        virtual TBase::TResult DoParseResult(const NNeh::THttpReply& report) const override {
            NJson::TJsonValue rawResult;
            if (!NJson::ReadJsonFastTree(report.Content(), &rawResult)) {
                ythrow yexception() << "Error getting objects by callId(" << CallId << "): " << report.GetDebugReply();
            }
            return rawResult;
        }
    };

    class TLoadTrackApplier : public IApplier<TString> {
        using TBase = IApplier<TString>;

    public:
        TLoadTrackApplier()
            : TBase("get-record")
        {
        }

    private:
        virtual TBase::TResult DoParseResult(const NNeh::THttpReply& report) const override {
            return report.Content();
        }
    };
}

namespace NDrive {

    TSupportAIClient::TSupportAIClient(const TSupportAIApiConfig& config, TAtomicSharedPtr<NTvmAuth::TTvmClient> tvmClient)
        : Config(config)
        , Client(MakeAtomicShared<NNeh::THttpClient>(Config.GetUri(), Config.GetRequestConfig()))
        , TvmAuth(tvmClient && config.GetDestinationTvmId() ? MakeMaybe<NDrive::TTvmAuth>(tvmClient, config.GetDestinationTvmId()) : Nothing())
    {
    }

    const TSupportAIApiConfig& TSupportAIClient::GetConfig() const {
        return Config;
    }

    NThreading::TFuture<TSupportAICall> TSupportAIClient::MakeCall(const TString& phone, const TMap<TString, TString>& features) const {
        NNeh::THttpRequest request;
        request.SetUri(Config.GetCallPath());
        request.SetCgiData("project_slug=" + Config.GetProjectSlug());
        NJson::TJsonValue featuresJson = NJson::JSON_ARRAY;
        for (auto&& [key, val] : features) {
            featuresJson.AppendValue(NJson::TMapBuilder("key", key)("value", val));
        }
        request.SetPostData(NJson::TMapBuilder("phone", phone)("features", featuresJson));
        if (TvmAuth) {
            TvmAuth->UpdateRequest(request);
        }
        return Client->SendAsync(request, Now() + Config.GetRequestTimeout()).Apply(TMakeCallApplier(phone));
    }

    NThreading::TFuture<NJson::TJsonValue> TSupportAIClient::GetCallResult(const TString& callId) const {
        NNeh::THttpRequest request;
        request.SetUri(Config.GetCallResultPath());
        request.SetCgiData("project_slug=" + Config.GetProjectSlug());
        NJson::TJsonValue callIds = NJson::JSON_ARRAY;
        callIds.AppendValue(callId);
        request.SetPostData(NJson::TMapBuilder("call_ids", callIds));
        if (TvmAuth) {
            TvmAuth->UpdateRequest(request);
        }
        return Client->SendAsync(request, Now() + Config.GetRequestTimeout()).Apply(TCallResultApplier(callId));
    }

    NThreading::TFuture<TString> TSupportAIClient::LoadTrack(const TString& callId) const {
        NNeh::THttpRequest request;
        TString path = Config.GetCallRecordPath();
        SubstGlobal(path, "__call_id__", callId);
        request.SetUri(path);
        request.SetCgiData("project_slug=" + Config.GetProjectSlug());
        if (TvmAuth) {
            TvmAuth->UpdateRequest(request);
        }
        return Client->SendAsync(request, Now() + Config.GetRequestTimeout()).Apply(TLoadTrackApplier());
    }
}
