#include "client.h"

#include <rtline/util/network/http_request.h>

#include <library/cpp/http/multipart/send.h>
#include <library/cpp/scheme/scheme.h>

#include <util/generic/set.h>
#include <util/generic/string.h>
#include <util/generic/noncopyable.h>

bool TTakeoutClient::SendFileToTakeout(const TString& jobId, const TString& filename, const TString& data) const {
    THttpHeaders headers;
    headers.AddHeader("X-Ya-Service-Ticket", Tvm->GetServiceTicketFor(Config.GetDestinationTvmId()));

    TString requestData = BuildMultipartHttpRequest(
        TVector<TStringPair>(
            {
                std::make_pair("job_id", jobId),
                std::make_pair("file\"; filename=\"" + filename, data)
            }
        ),
        "/1/upload/?consumer=" + Config.GetConsumer(),
        Config.GetHost(),
        DefaultBoundary,
        headers
    );

    return SendHttpRequest(requestData);
}

bool TTakeoutClient::FinalizeReport(const TString& jobId, const TVector<TString>& filenames) const {
    THttpHeaders headers;
    headers.AddHeader("X-Ya-Service-Ticket", Tvm->GetServiceTicketFor(Config.GetDestinationTvmId()));

    TVector<TStringPair> contents;
    contents.reserve((filenames.size() + 1));

    contents.push_back(std::make_pair("job_id", jobId));
    for (auto&& element : filenames) {
        contents.push_back(std::make_pair("filename", element));
    }

    TString requestData = BuildMultipartHttpRequest(
        contents,
        "/1/upload/done/?consumer=" + Config.GetConsumer(),
        Config.GetHost(),
        DefaultBoundary,
        headers
    );

    return SendHttpRequest(requestData);
}

bool TTakeoutClient::SendHttpRequest(const TString& rawPayload) const {
    NUtil::THttpReply sendResult;
    try {
        bool isSucceeded = NRTLine::SendRequest(Config.GetHost(), Config.GetPort(), rawPayload, Config.GetRequestTimeout(), sendResult, true);

        if (isSucceeded) {
            isSucceeded = sendResult.Code() / 100 == 2;
        }

        if (isSucceeded) {
            NSc::TValue content = NSc::TValue::FromJson(sendResult.Content());
            isSucceeded = (content["status"] == "ok" || (content["status"] == "job_id.invalid" && content["error"] == "Task is complete"));
        }

        if (!isSucceeded) {
            ERROR_LOG << "TAKEOUT CLIENT REQUEST FAILED" << Endl << sendResult.Code() << " " << sendResult.ErrorMessage() << Endl << sendResult.Content() << Endl;
        } else if (Config.GetDebugOutput()) {
            Cerr << sendResult.Code() << Endl << sendResult.Content() << Endl;
        }

        return isSucceeded;
    } catch (const std::exception& e) {
        ERROR_LOG << "TAKEOUT CLIENT EXCEPTION: " << FormatExc(e) << Endl;
        return false;
    }
}

bool TFakeTakeoutClient::SendFileToTakeout(const TString& jobId, const TString& filename, const TString& data) const {
    UploadedFiles[jobId].push_back(std::make_pair(filename, data));
    return true;
}

bool TFakeTakeoutClient::FinalizeReport(const TString& jobId, const TVector<TString>& filenames) const {
    if (filenames.size() != UploadedFiles[jobId].size()) {
        return false;
    }

    TSet<TString> uploadedFilenames;
    for (auto&& element : UploadedFiles[jobId]) {
        uploadedFilenames.insert(element.first);
    }

    for (auto&& filename : filenames) {
        if (!uploadedFilenames.contains(filename)) {
            return false;
        }
    }

    return true;
}

TVector<TFakeTakeoutClient::TStringPair> TFakeTakeoutClient::GetArchiveContents(const TString& jobId) const {
    return UploadedFiles[jobId];
}
