#include "client.h"

#include <drive/backend/logging/events.h>

#include <drive/library/cpp/threading/future.h>
#include <rtline/library/json/parse.h>

#include <library/cpp/json/json_reader.h>
#include <util/string/builder.h>

template <>
NJson::TJsonValue NJson::ToJson(const TTaxiAntifraudClient::TFacesSimilarityInfo& info) {
    NJson::TJsonValue result;
    NJson::InsertField(result, "found_face_on_photo1", info.GetFoundFaceOnFirstPhoto());
    NJson::InsertField(result, "found_face_on_photo2", info.GetFoundFaceOnSecondPhoto());
    NJson::InsertField(result, "similarity", ToString(info.GetSimilarity()));
    return result;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TTaxiAntifraudClient::TFacesSimilarityInfo& result) {
    return
        NJson::ParseField(value, "found_face_on_photo1", result.MutableFoundFaceOnFirstPhoto(), true) &&
        NJson::ParseField(value, "found_face_on_photo2", result.MutableFoundFaceOnSecondPhoto(), true) &&
        NJson::ParseField(value, "similarity", result.MutableSimilarity(), true);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TTaxiAntifraudClient::TRecognizedText& result) {
    return
        NJson::ParseField(value, "type", result.Type, false) &&
        NJson::ParseField(value, "text", result.Text, false) &&
        NJson::ParseField(value, "confidence", result.Confidence, false);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TTaxiAntifraudClient::TRecognizedDocument& result) {
    return
        NJson::ParseField(value, "recognized_text", result.MutableRecognizedTexts(), false) &&
        NJson::ParseField(value, "quasi_fms_score", result.MutableQuasiFmsScore(), false) &&
        NJson::ParseField(value, "quasi_gibdd_score", result.MutableQuasiGibddScore(), false) &&
        NJson::ParseField(value, "from_screen_model_score", result.MutableFromScreenScore(), false) &&
        NJson::ParseField(value, "bad_format_model_score", result.MutableBadFormatScore(), false);
}

NThreading::TFuture<TTaxiAntifraudClient::TFacesSimilarityInfo> TTaxiAntifraudClient::CalculateFacesSimilarity(const TString& userId, const TString& photo1Blob, const TString& photo2Blob) const {
    if (!TvmClient) {
        return NThreading::MakeErrorFuture<TTaxiAntifraudClient::TFacesSimilarityInfo>(std::make_exception_ptr(yexception() << "TTaxiAntifraudClient::CalculateFacesSimilarity TvmClient not configured"));
    }

    NNeh::THttpRequest request;
    TStringBuilder uriBuilder;
    uriBuilder << Config.GetFacesSimilarityPath();
    TVector<NNeh::THttpRequest::TMultipartData> parts = {
        {"photo1", "photo1.jpg", "image/jpeg", photo1Blob},
        {"photo2", "photo2.jpg", "image/jpeg", photo2Blob}
    };
    request.SetUri(uriBuilder)
        .SetRequestType("POST")
        .AddHeader("X-Ya-Service-Ticket", TvmClient->GetServiceTicketFor(Config.GetDestinationTvmId()));
    request.SetMultipartPostData(parts);
    NDrive::TEventLog::Log("TTaxiAntifraudClient::CalculateFacesSimilarity", NJson::TMapBuilder
        ("user_id", userId)
        ("request", request.Serialize())
        ("uri", uriBuilder)
    );

    return Agent->SendAsync(request, Now() + Config.GetRequestTimeout()).Apply([userId](const NThreading::TFuture<NUtil::THttpReply>& r) {
        if (r.HasException()) {
            throw NThreading::GetException(r);
        }
        if (!r.HasValue()) {
            return NThreading::MakeErrorFuture<TFacesSimilarityInfo>(std::make_exception_ptr(yexception() << "No value in http reply future"));
        }
        const auto& reply = r.GetValue();
        if (!reply.IsSuccessReply()) {
            return NThreading::MakeErrorFuture<TFacesSimilarityInfo>(std::make_exception_ptr(yexception() << "Request error, reply code " << reply.Code() << ", error: " << reply.ErrorMessage() << ", content: " << reply.Content()));
        };

        NJson::TJsonValue replyJson;
        if (!NJson::ReadJsonFastTree(reply.Content(), &replyJson)) {
            return NThreading::MakeErrorFuture<TFacesSimilarityInfo>(std::make_exception_ptr(yexception() << "Cannot parse reply json " << reply.Content()));
        }
        NDrive::TEventLog::Log("TTaxiAntifraudClient::CalculateFacesSimilarity", NJson::TMapBuilder
            ("user_id", userId)
            ("reply", replyJson)
        );
        TTaxiAntifraudClient::TFacesSimilarityInfo similarityInfo;
        if (!TryFromJson(replyJson, similarityInfo)) {
            return NThreading::MakeErrorFuture<TFacesSimilarityInfo>(std::make_exception_ptr(yexception() << "Cannot suggest data from reply"));
        }
        return NThreading::MakeFuture(similarityInfo);
    });
}

NThreading::TFuture<TTaxiAntifraudClient::TRecognizedDocument> TTaxiAntifraudClient::SendPhoto(const TString& photoBlob, const TString& uri) const {
    if (!TvmClient) {
        return NThreading::TExceptionFuture() << "TvmClient not configured";
    }
    NNeh::THttpRequest request;
    TVector<NNeh::THttpRequest::TMultipartData> parts = {
        {"photo", "photo.jpg", "image/jpeg", photoBlob}
    };
    request.SetUri(uri)
        .SetRequestType("POST")
        .AddHeader("X-Ya-Service-Ticket", TvmClient->GetServiceTicketFor(Config.GetDestinationTvmId()));
    NDrive::TEventLog::Log("TTaxiAntifraudClient::SendPhoto", NJson::TMapBuilder
        ("request", request.Serialize())
    );
    request.SetMultipartPostData(parts);

    return Agent->SendAsync(request, Now() + Config.GetRequestTimeout()).Apply([](const NThreading::TFuture<NUtil::THttpReply>& r) -> NThreading::TFuture<TTaxiAntifraudClient::TRecognizedDocument> {
        if (r.HasException()) {
            return NThreading::TExceptionFuture() << NThreading::GetExceptionMessage(r);
        }
        if (!r.HasValue()) {
            return NThreading::TExceptionFuture() << "No value in http reply future";
        }
        const auto& reply = r.GetValue();
        if (!reply.IsSuccessReply()) {
            return NThreading::TExceptionFuture() << "Request error, reply code " << reply.Code() << ", error: " << reply.ErrorMessage() << ", content: " << reply.Content();
        };
        NJson::TJsonValue replyJson;
        if (!NJson::ReadJsonFastTree(reply.Content(), &replyJson)) {
            return NThreading::TExceptionFuture() << "Cannot parse reply json " << reply.Content();
        }
        TTaxiAntifraudClient::TRecognizedDocument result;
        TString error;
        if (!NJson::ParseField(replyJson, result, &error)) {
            return NThreading::TExceptionFuture() << "Cannot parse fields from json: " << error;
        }
        return NThreading::MakeFuture(result);
    });
}

NThreading::TFuture<TTaxiAntifraudClient::TRecognizedDocument> TTaxiAntifraudClient::RecognizePassport(const TString& photoBlob) const {
    return SendPhoto(photoBlob, Config.GetRecognizePassportPath());
}

NThreading::TFuture<TTaxiAntifraudClient::TRecognizedDocument> TTaxiAntifraudClient::RecognizeLicenseFront(const TString& photoBlob) const {
    return SendPhoto(photoBlob, Config.GetRecognizeLicenseFrontPath());
}

NThreading::TFuture<TTaxiAntifraudClient::TRecognizedDocument> TTaxiAntifraudClient::RecognizeLicenseBack(const TString& photoBlob) const {
    return SendPhoto(photoBlob, Config.GetRecognizeLicenseBackPath());
}
