#pragma once

#include <util/stream/str.h>
#include <util/generic/cast.h>
#include <util/generic/map.h>

#include <library/cpp/neh/neh.h>
#include <library/cpp/http/misc/httpcodes.h>
#include <library/cpp/logger/global/global.h>

namespace NRTLine {
    class TAsyncReplyBase {
    public:
        TAsyncReplyBase(ui16 code, const TString& message)
            : RawReport(message)
            , Code(code)
            , Fetched(true)
            , Succeded(code == HTTP_OK)
        {}

        TAsyncReplyBase(NNeh::THandleRef handle, TDuration timeout)
            : Handle(handle)
            , Timeout(timeout)
        {
            Y_VERIFY(Handle);
        }

        bool IsSucceeded() const {
            Fetch();
            return Succeded;
        }

        i32 GetCode() const {
            Fetch();
            return Code;
        }

        const TString& GetRawReport() const {
            Fetch();
            return RawReport;
        }

        virtual bool ParseRawReply(const TString& /*rawData*/) const{
            return true;
        }

        virtual ~TAsyncReplyBase() {}

    protected:
        void Fetch() const {
            auto guard = Guard(Lock);
            if (Fetched) {
                return;
            }

            try {
                const NNeh::TResponseRef response = Handle->Wait(Timeout);
                if (response) {
                    RawReport = response->Data;
                    if (response->IsError()) {
                        Code = response->GetErrorCode();
                        if (!RawReport) {
                            RawReport = response->GetErrorText();
                        }
                    } else {
                        Code = HTTP_OK;
                        Succeded = ParseRawReply(RawReport);
                    }
                } else {
                    Code = HTTP_GATEWAY_TIME_OUT;
                }
            } catch (const std::exception& e) {
                Code = HTTP_INTERNAL_SERVER_ERROR;
                RawReport = FormatExc(e);
            }

            Fetched = true;
        }

    private:
        const NNeh::THandleRef Handle;
        const TDuration Timeout;

        mutable TAdaptiveLock Lock;

        mutable TString RawReport;
        mutable i32 Code = -1;
        mutable bool Fetched = false;
        mutable bool Succeded = false;
    };
}

namespace NTaxi {
    class THttpClient {
    public:
        class IParsedReport {
        public:
            using TPtr = TAtomicSharedPtr<IParsedReport>;
            virtual ~IParsedReport() {}
        };

        class IReplyParser {
        public:
            using TPtr = TAtomicSharedPtr<IReplyParser>;
            virtual ~IReplyParser() {}
            virtual IParsedReport::TPtr Parse(const TString& rawData) const = 0;
        };

        class TReply: public NRTLine::TAsyncReplyBase {
        public:
            using TPtr = TAtomicSharedPtr<TReply>;
            TReply(const TString& error)
                : TAsyncReplyBase(HTTP_INTERNAL_SERVER_ERROR, error)
            {}

            TReply(NNeh::THandleRef handle, TDuration timeout, IReplyParser::TPtr parser)
                : NRTLine::TAsyncReplyBase(handle, timeout)
                , Parser(parser)
            {
                CHECK_WITH_LOG(Parser);
            }

            virtual bool ParseRawReply(const TString& rawData) const override {
                ParsedReport = Parser->Parse(rawData);
                return !!ParsedReport.Get();
            }

            template <class TParsedReport>
            const TParsedReport& GetReport() const {
                return *VerifyDynamicCast<TParsedReport*>(ParsedReport.Get());
            }

        private:
            mutable IParsedReport::TPtr ParsedReport;
            IReplyParser::TPtr Parser;
        };

    protected:
        TString MakeFullHttp(const TString& host, ui16 port, const TString& request, const ::TMap<TString, TString>& headers, const TString& method,  const TString& data = "") const {
            TStringStream ss;
            TStringBuf CrLf = "\r\n";
            ss << method << " /"sv;
            ss << request << " HTTP/1.1"sv << CrLf;
            ss << "Host: "sv << host << ":"sv << port;
            for (auto&& header : headers) {
                ss << CrLf << header.first << ": "sv << header.second;
            }

            if (data) {
                ss << CrLf << "Content-Length: "sv << data.size();
                ss << CrLf << CrLf;
                ss << data;
            } else {
                ss << CrLf << CrLf;
            }

            return ss.Str();
        }
    };
}
