#include "client.h"

#include <drive/library/cpp/threading/future.h>

#include <rtline/util/json_processing.h>

#include <util/string/builder.h>

THolder<ITaxiChatClient> ITaxiChatClient::ConstructTaxiChatClient(const TTaxiChatClientConfig& config, TAtomicSharedPtr<NTvmAuth::TTvmClient> tvmClient) {
    if (config.GetClientType() == "test") {
        return MakeHolder<TTestTaxiChatClient>();
    } else if (tvmClient) {
        return MakeHolder<TTaxiChatClient>(config, tvmClient);
    } else {
        return nullptr;
    }
}

NThreading::TFuture<bool> ProcessReply(const TString& endpointPath, const NThreading::TFuture<NUtil::THttpReply>& r) {
    if (r.HasException()) {
        throw NThreading::GetException(r);
    }
    if (!r.HasValue()) {
        return NThreading::MakeErrorFuture<bool>(std::make_exception_ptr(yexception() << "No value in http reply future"));
    }
    const auto& reply = r.GetValue();
    if (!reply.IsSuccessReply()) {
        return NThreading::MakeErrorFuture<bool>(std::make_exception_ptr(yexception() << endpointPath << " request error, reply code " << reply.Code() << ", error: " << reply.ErrorMessage() << ", content: " << reply.Content()));
    }
    return NThreading::MakeFuture(true);
}

TExpected<NNeh::THttpRequest, TString> TTaxiChatClient::CreateRequest(const TString& chatId, const TString& endpoint, const TMap<TString, TString>& queryParams, const NJson::TJsonValue& postData /*= NJson::JSON_NULL*/) const {
    if (!TvmClient) {
        return MakeUnexpected<TString>("TvmClient not configured");
    }
    NNeh::THttpRequest request;
    TStringBuilder uriBuilder;
    uriBuilder << Config.GetTaskPathPrefix()
                << chatId << "/"
                << endpoint << "/?"
                << "task_id="
                << chatId;
    for (const auto& [key, value] : queryParams) {
        uriBuilder << "&" << key << "=" << value;
    }
    request.SetUri(uriBuilder)
        .SetRequestType("POST")
        .AddHeader("Content-Type", "application/json")
        .AddHeader("X-Ya-Service-Ticket", TvmClient->GetServiceTicketFor(Config.GetDestinationTvmId()));
    if (postData.IsDefined()) {
        request.SetPostData(postData);
    }
    INFO_LOG << "TaxiChatClient: " << request.GetDebugRequest() << Endl;
    return request;
}

NThreading::TFuture<bool> TTaxiChatClient::SetCommentAndSend(const TString& externalChatId, const TString& endpointPath, const TMaybe<TString>& comment, const TMaybe<TString>& hiddenComment) const {
    NJson::TJsonValue postData = NJson::JSON_MAP;
    if (comment) {
        postData.InsertValue("comment", *comment);
    }
    if (hiddenComment) {
        postData.InsertValue("hidden_comment", *hiddenComment);
    }
    auto request = CreateRequest(externalChatId, endpointPath, {}, postData);
    if (request) {
        return Agent->SendAsync(request.GetRef(), Now() + Config.GetRequestTimeout()).Apply([endpointPath](const NThreading::TFuture<NUtil::THttpReply>& r) {
            return ProcessReply(endpointPath, r);
        });
    } else {
        return NThreading::MakeErrorFuture<bool>(std::make_exception_ptr(yexception() << request.GetError()));
    }
}

NThreading::TFuture<bool> TTaxiChatClient::AddMessage(const TString& externalChatId, const TString& message) const {
    return SetCommentAndSend(externalChatId, Config.GetAddMessagePath(), message);
}

NThreading::TFuture<bool> TTaxiChatClient::DeferWithMessage(const TString& externalChatId, const TString& message, const TMaybe<TString>& comment) const {
    return SetCommentAndSend(externalChatId, Config.GetDeferWithMessagePath(), message, comment);
}

NThreading::TFuture<bool> TTaxiChatClient::CloseChatWithMessage(const TString& externalChatId, const TString& message) const {
    return SetCommentAndSend(externalChatId, Config.GetCloseChatPath(), message);
}

NThreading::TFuture<bool> TTaxiChatClient::CloseChat(const TString& externalChatId, const TString& comment /*= ""*/) const {
    return SetCommentAndSend(externalChatId, Config.GetDismissChatPath(), {}, comment);
}

NThreading::TFuture<bool> TTaxiChatClient::ForwardMessages(const TString& externalChatId, const TString& newChatType, const TVector<TString>& messageExternalIds, const TString& comment /*= ""*/) const {
    NJson::TJsonValue postData = NJson::JSON_MAP;
    if (comment) {
        postData.InsertValue("hidden_comment", comment);
    }
    TJsonProcessor::WriteContainerArray(postData, "selected_messages_id", messageExternalIds);
    TString endpointPath = Config.GetForwardMessagesPath();
    auto request = CreateRequest(externalChatId, Config.GetForwardMessagesPath(), {{"new_chat_type", newChatType}}, postData);
    if (request) {
        return Agent->SendAsync(request.GetRef(), Now() + Config.GetRequestTimeout()).Apply([endpointPath](const NThreading::TFuture<NUtil::THttpReply>& r) {
            return ProcessReply(endpointPath, r);
        });
    } else {
        return NThreading::MakeErrorFuture<bool>(std::make_exception_ptr(yexception() << request.GetError()));
    }
}

ITaxiChatClient::TTaxiChatAttachment::TTaxiChatAttachment(const NUtil::THttpReply& httpReply, const TString& attachmentId, const TString& externalMessageId)
    : Id(attachmentId)
    , ExternalMessageId(externalMessageId)
{
    ContentType = "image/jpeg";
    auto contentHeader = httpReply.GetHeaders().FindHeader("Content-Type");
    if (contentHeader) {
        ContentType = contentHeader->Value();
    }
    Content = httpReply.Content();
}

NThreading::TFuture<ITaxiChatClient::TTaxiChatAttachment> TTaxiChatClient::GetAttachment(const TString& externalChatId, const TString& attachmentId, const TString& externalMessageId) const {
    if (!TvmClient) {
        return NThreading::MakeErrorFuture<TTaxiChatClient::TTaxiChatAttachment>(std::make_exception_ptr(yexception() << "TvmClient not configured"));
    }
    NNeh::THttpRequest request;
    TStringBuilder uriBuilder;
    uriBuilder << Config.GetTaskPathPrefix()
                << externalChatId << "/"
                << Config.GetAttachmentPath() << "/"
                << attachmentId;
    request.SetUri(uriBuilder)
        .SetRequestType("GET")
        .AddHeader("X-Ya-Service-Ticket", TvmClient->GetServiceTicketFor(Config.GetDestinationTvmId()));
    INFO_LOG << "TaxiChatClient: " << request.GetDebugRequest() << Endl;
    return Agent->SendAsync(request, Now() + Config.GetRequestTimeout()).Apply([attachmentId, externalMessageId](const NThreading::TFuture<NUtil::THttpReply>& r) {
        if (r.HasException()) {
            throw NThreading::GetException(r);
        }
        if (!r.HasValue()) {
            throw NThreading::MakeErrorFuture<ITaxiChatClient::TTaxiChatAttachment>(std::make_exception_ptr(yexception() << "No value in http reply future"));
        }
        const auto& reply = r.GetValue();
        if (reply.Code() != 200) {
            throw yexception() << "Get attachment " << attachmentId << " request error, reply code " << reply.Code() << ", error: " << reply.ErrorMessage() << ", content: " << reply.Content();
        } else {
            return ITaxiChatClient::TTaxiChatAttachment(reply, attachmentId, externalMessageId);
        }
    });
}

NThreading::TFuture<bool> TTestTaxiChatClient::CloseChat(const TString& /*externalChatId*/, const TString& /*comment*/) const {
    return ProcessReply("close_chat", NThreading::MakeFuture(Reply));
}

NThreading::TFuture<bool> TTestTaxiChatClient::CloseChatWithMessage(const TString& /*externalChatId*/, const TString& /*message*/) const {
    return ProcessReply("close_chat_with_message", NThreading::MakeFuture(Reply));
}

NThreading::TFuture<bool> TTestTaxiChatClient::AddMessage(const TString& /*externalChatId*/, const TString& /*message*/) const {
    return ProcessReply("add_message", NThreading::MakeFuture(Reply));
}

NThreading::TFuture<bool> TTestTaxiChatClient::DeferWithMessage(const TString& /*externalChatId*/, const TString& /*message*/, const TMaybe<TString>& /*comment*/) const {
    return ProcessReply("defer_with_message", NThreading::MakeFuture(Reply));
}

NThreading::TFuture<bool> TTestTaxiChatClient::ForwardMessages(const TString& /*externalChatId*/, const TString& /*newChatType*/, const TVector<TString>& /*messageExternalIds*/, const TString& /*comment = ""*/) const {
    return ProcessReply("forward_messages", NThreading::MakeFuture(Reply));
}

NThreading::TFuture<ITaxiChatClient::TTaxiChatAttachment> TTestTaxiChatClient::GetAttachment(const TString& /*externalChatId*/, const TString& attachmentId, const TString& externalMessageId) const {
    auto reply = AttachmentsReply[attachmentId];
    reply.SetExternalMessageId(externalMessageId);
    return NThreading::MakeFuture(reply);
}
