#pragma once

#include "config.h"

#include <rtline/util/network/neh.h>
#include <rtline/util/types/expected.h>
#include <library/cpp/tvmauth/client/facade.h>

class ITaxiChatClient {
public:
    class TTaxiChatAttachment {
        R_FIELD(TString, Content);
        R_FIELD(TString, ContentType);
        R_FIELD(TString, Id);
        R_FIELD(TString, ExternalMessageId);

    public:
        TTaxiChatAttachment() = default;
        TTaxiChatAttachment(const NUtil::THttpReply& httpReply, const TString& attachmentId, const TString& externalMessageId);
    };

public:
    virtual ~ITaxiChatClient() = default;

    virtual NThreading::TFuture<bool> CloseChat(const TString& externalChatId, const TString& comment = "") const = 0;
    virtual NThreading::TFuture<bool> CloseChatWithMessage(const TString& externalChatId, const TString& message) const = 0;
    virtual NThreading::TFuture<bool> AddMessage(const TString& externalChatId, const TString& message) const = 0;
    virtual NThreading::TFuture<bool> DeferWithMessage(const TString& externalChatId, const TString& message, const TMaybe<TString>& comment = {}) const = 0;
    virtual NThreading::TFuture<bool> ForwardMessages(const TString& externalChatId, const TString& newChatType, const TVector<TString>& messageExternalIds, const TString& comment = "") const = 0;
    virtual NThreading::TFuture<ITaxiChatClient::TTaxiChatAttachment> GetAttachment(const TString& externalChatId, const TString& attachmentId, const TString& externalMessageId) const = 0;

    static THolder<ITaxiChatClient> ConstructTaxiChatClient(const TTaxiChatClientConfig& config, TAtomicSharedPtr<NTvmAuth::TTvmClient> tvmClient);
};

class TTaxiChatClient: public ITaxiChatClient {
public:
    TTaxiChatClient(const TTaxiChatClientConfig& config, TAtomicSharedPtr<NTvmAuth::TTvmClient> tvmClient)
        : Config(config)
        , TvmClient(tvmClient)
    {
        CHECK_WITH_LOG(tvmClient);
        Agent = MakeHolder<NNeh::THttpClient>(Config.GetUri(), Config.GetRequestConfig());
    }

    virtual NThreading::TFuture<bool> CloseChat(const TString& externalChatId, const TString& comment = "") const override;
    virtual NThreading::TFuture<bool> CloseChatWithMessage(const TString& externalChatId, const TString& message) const override;
    virtual NThreading::TFuture<bool> AddMessage(const TString& externalChatId, const TString& message) const override;
    virtual NThreading::TFuture<bool> DeferWithMessage(const TString& externalChatId, const TString& message, const TMaybe<TString>& comment = {}) const override;
    virtual NThreading::TFuture<bool> ForwardMessages(const TString& externalChatId, const TString& newChatType, const TVector<TString>& messageExternalIds, const TString& comment = "") const override;
    virtual NThreading::TFuture<ITaxiChatClient::TTaxiChatAttachment> GetAttachment(const TString& externalChatId, const TString& attachmentId, const TString& externalMessageId) const override;

private:
    NThreading::TFuture<bool> SetCommentAndSend(const TString& externalChatId, const TString& endpointPath, const TMaybe<TString>& comment = {}, const TMaybe<TString>& hiddenComment = {}) const;
    TExpected<NNeh::THttpRequest, TString> CreateRequest(const TString& chatId, const TString& endpoint, const TMap<TString, TString>& queryParams, const NJson::TJsonValue& postData = NJson::JSON_NULL) const;

private:
    THolder<NNeh::THttpClient> Agent;
    const TTaxiChatClientConfig& Config;
    TAtomicSharedPtr<NTvmAuth::TTvmClient> TvmClient;
};

class TTestTaxiChatClient: public ITaxiChatClient {
private:
    using AttachmentsMap = TMap<TString, ITaxiChatClient::TTaxiChatAttachment>;
    R_FIELD(NUtil::THttpReply, Reply, {}, mutable);
    R_FIELD(AttachmentsMap, AttachmentsReply, {}, mutable);

public:
    TTestTaxiChatClient() = default;

    virtual NThreading::TFuture<bool> CloseChat(const TString& externalChatId, const TString& comment = "") const override;
    virtual NThreading::TFuture<bool> CloseChatWithMessage(const TString& externalChatId, const TString& message) const override;
    virtual NThreading::TFuture<bool> AddMessage(const TString& externalChatId, const TString& message) const override;
    virtual NThreading::TFuture<bool> DeferWithMessage(const TString& externalChatId, const TString& message, const TMaybe<TString>& comment = {}) const override;
    virtual NThreading::TFuture<bool> ForwardMessages(const TString& externalChatId, const TString& newChatType, const TVector<TString>& messageExternalIds, const TString& comment = "") const override;
    virtual NThreading::TFuture<ITaxiChatClient::TTaxiChatAttachment> GetAttachment(const TString& externalChatId, const TString& attachmentId, const TString& externalMessageId) const override;
};
