#include "client.h"

#include "library/cpp/logger/global/global.h"

#include <util/charset/utf8.h>
#include <util/string/builder.h>

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TTaxiDriverProfilesClient::TRecognizedParkDriverProfileId& result) {
    TVector<TString> parkIdDriverProfileId = StringSplitter(value.GetStringRobust()).SplitBySet("_").SkipEmpty().ToList<TString>();
    if (parkIdDriverProfileId.size() != 2) {
        return false;
    }
    result.parkId = parkIdDriverProfileId[0];
    result.driverProfileId = parkIdDriverProfileId[1];
    return true;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TTaxiDriverProfilesClient::TRecognizedFullName& result) {
    return
        NJson::ParseField(value, "first_name", result.FirstName, false) &&
        NJson::ParseField(value, "middle_name", result.MiddleName, false) &&
        NJson::ParseField(value, "last_name", result.LastName, false);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TTaxiDriverProfilesClient::TRecognizedData& result) {
    return
        NJson::ParseField(value, "full_name", result.MutableFullName(), false) &&
        NJson::ParseField(value, "fire_date", result.MutableFireDate(), false) &&
        NJson::ParseField(value, "hire_date", result.MutableHireDate(), false) &&
        NJson::ParseField(value, "work_status", NJson::Stringify(result.MutableWorkStatus()), false);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TTaxiDriverProfilesClient::TRecognizedProfile& result) {
    return
        NJson::ParseField(value, "park_driver_profile_id", result.MutableParkDriverProfileId(), true) &&
        NJson::ParseField(value, "revision", result.MutableRevision(), false) &&
        NJson::ParseField(value, "is_deleted", result.MutableIsDeleted(), false) &&
        NJson::ParseField(value, "data", result.MutableData(), false);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TTaxiDriverProfilesClient::TRecognizedProfilesByParkIdCarId& result) {
    return
        NJson::ParseField(value, "park_id_car_id", result.MutableParkIdCarId(), false) &&
        NJson::ParseField(value, "profiles", result.MutableProfiles(), false);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TTaxiDriverProfilesClient::TRecognizedProfilesByParkIdCarIds& result) {
    return
        NJson::ParseField(value, "profiles_by_park_id_car_id", result.MutableRecognizedProfilesByParkIdCarId(), false);
}

NJson::TJsonValue TTaxiDriverProfilesClient::TRequestParams::ToJson() const {
    NJson::TJsonValue result = NJson::JSON_MAP;
    result.InsertValue("park_id_car_id_in_set", NJson::ToJson(GetParkIdCarIdInSet()));
    if (GetProjection().size()) {
        result.InsertValue("projection", NJson::ToJson(GetProjection()));
    }
    return result;
}

NThreading::TFuture<TTaxiDriverProfilesClient::TRecognizedProfilesByParkIdCarIds> TTaxiDriverProfilesClient::GetParkDriverProfileIds(const TRequestParams& requestParams, TAtomicSharedPtr<NDrive::TEventLog::TState> eventLogState) const {
    if (!Tvm) {
        throw yexception() << "TTaxiDriverProfilesClient::TRecognizedProfilesByParkIdCarIds TvmClient not configured";
    }
    auto req = CreateRequest();
    req.SetUri(Config.GetEndpointParkIdCarId());
    req.SetPostData(requestParams.ToJson().GetStringRobust());

    return Client->SendAsync(req, Now() + Config.GetRequestTimeout())
        .Apply([eventLogState = std::move(eventLogState)](const NThreading::TFuture<NNeh::THttpReply>& response) {
            const auto& reply = response.GetValue();
            if (eventLogState) {
                NDrive::TEventLog::TStateGuard stateGuard(*eventLogState);
                if (response.HasValue()) {
                    NDrive::TEventLog::Log("TTaxiDriverProfilesClient", NJson::TMapBuilder
                        ("code", reply.Code())
                        ("response", reply.Content())
                    );
                }
            }
            reply.EnsureSuccessfulReply();
            NJson::TJsonValue replyJson;
            if (!NJson::ReadJsonFastTree(reply.Content(), &replyJson)) {
                throw yexception() << "Cannot parse reply json " << reply.Content();
            }
            TRecognizedProfilesByParkIdCarIds recognizedProfilesByParkIdCarIds;
            if (!TryFromJson(replyJson, recognizedProfilesByParkIdCarIds)) {
                throw yexception() << "Cannot suggest data from reply";
            }
            return NThreading::MakeFuture(recognizedProfilesByParkIdCarIds);
        }
    );
}

NNeh::THttpRequest TTaxiDriverProfilesClient::CreateRequest() const {
    NNeh::THttpRequest result;
    result.AddHeader("Content-Type", "application/json")
        .AddHeader("X-Ya-Service-Ticket", Tvm->GetServiceTicketFor(Config.GetDestinationTvmId()));

    auto cgiData = result.GetCgiData();
    result.SetCgiData(
        TStringBuilder()
        << cgiData
        << ((cgiData) ? "&" : "")
        << "consumer=" << Config.GetConsumer()
    );
    return result;
}

TTaxiDriverProfilesClient::TTaxiDriverProfilesClient(const TTaxiDriverProfilesConfig& config, TAtomicSharedPtr<NTvmAuth::TTvmClient> tvm)
    : Config(config)
    , Tvm(tvm)
{
    Client = MakeHolder<NNeh::THttpClient>(Config.GetUri(), Config.GetRequestConfig());
}
