#include "client.h"

#include "library/cpp/logger/global/global.h"

#include <util/charset/utf8.h>
#include <util/string/builder.h>

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TTaxiDriverStatusClient::TRecognizedOrder& result) {
    return
        NJson::ParseField(value, "id", result.MutableId(), false) &&
        NJson::ParseField(value, "status", result.MutableStatus(), false);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TTaxiDriverStatusClient::TRecognizedStatus& result) {
    return
        NJson::ParseField(value, "park_id", result.MutableParkId(), false) &&
        NJson::ParseField(value, "driver_id", result.MutableDriverId(), false) &&
        NJson::ParseField(value, "status", NJson::Stringify(result.MutableStatus()), false) &&
        NJson::ParseField(value, "updated_ts", result.MutableUpdatedTS(), false) &&
        NJson::ParseField(value, "orders", result.MutableOrders(), false);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TTaxiDriverStatusClient::TRecognizeStatuses& result) {
    return
        NJson::ParseField(value, "statuses", result.MutableRecognizedStatuses(), false);
}

NJson::TJsonValue TTaxiDriverStatusClient::TRequestParams::ToJson() const {
    NJson::TJsonValue result = NJson::JSON_MAP;
    NJson::TJsonValue& objectsJson =  result.InsertValue("driver_ids", NJson::JSON_ARRAY);
    for (const auto& parkIdDrivedId : GetParkIdDrivedIds()) {
        NJson::TJsonValue profile = NJson::JSON_MAP;
        profile.InsertValue("park_id", NJson::ToJson(parkIdDrivedId.ParkId));
        profile.InsertValue("driver_id", NJson::ToJson(parkIdDrivedId.DriverId));
        objectsJson.AppendValue(profile);
    }
    return result;
}

NThreading::TFuture<TTaxiDriverStatusClient::TRecognizeStatuses> TTaxiDriverStatusClient::GetDriversStatuses(const TRequestParams& requestParams, TAtomicSharedPtr<NDrive::TEventLog::TState> eventLogState) const {
    if (!Tvm) {
        throw yexception() << "TTaxiDriverStatusClient::TRecognizeStatuses TvmClient not configured";
    }
    auto req = CreateRequest();
    req.SetUri(Config.GetEndpoint());
    req.SetPostData(requestParams.ToJson().GetStringRobust());

    return Client->SendAsync(req, Now() + Config.GetRequestTimeout())
        .Apply([eventLogState = std::move(eventLogState)](const NThreading::TFuture<NNeh::THttpReply>& response) {
            const auto& reply = response.GetValue();
            if (eventLogState) {
                NDrive::TEventLog::TStateGuard stateGuard(*eventLogState);
                if (response.HasValue()) {
                    NDrive::TEventLog::Log("TTaxiDriverStatusClient", NJson::TMapBuilder
                        ("code", reply.Code())
                        ("response", reply.Content())
                    );
                }
            }
            reply.EnsureSuccessfulReply();
            NJson::TJsonValue replyJson;
            if (!NJson::ReadJsonFastTree(reply.Content(), &replyJson)) {
                throw yexception() << "Cannot parse reply json " << reply.Content();
            }
            TRecognizeStatuses recognizeStatuses;
            // recognizeStatuses.MutableRecognizedStatuses().clear();
            if (!TryFromJson(replyJson, recognizeStatuses)) {
                throw yexception() << "Cannot suggest data from reply";
            }
            return NThreading::MakeFuture(recognizeStatuses);
        }
    );
}

NNeh::THttpRequest TTaxiDriverStatusClient::CreateRequest() const {
    NNeh::THttpRequest result;
    result.AddHeader("Content-Type", "application/json")
        .AddHeader("X-Ya-Service-Ticket", Tvm->GetServiceTicketFor(Config.GetDestinationTvmId()));

    auto cgiData = result.GetCgiData();
    result.SetCgiData(
        TStringBuilder()
        << cgiData
        << ((cgiData) ? "&" : "")
        << "consumer=" << Config.GetConsumer()
    );
    return result;
}

TTaxiDriverStatusClient::TTaxiDriverStatusClient(const TTaxiDriverStatusConfig& config, TAtomicSharedPtr<NTvmAuth::TTvmClient> tvm)
    : Config(config)
    , Tvm(tvm)
{
    Client = MakeHolder<NNeh::THttpClient>(Config.GetUri(), Config.GetRequestConfig());
}
