#include "client.h"

#include <drive/library/cpp/threading/future.h>

#include <rtline/library/json/parse.h>

#include <util/string/builder.h>

NNeh::THttpRequest TTaxiTrackStoryClient::CreateRequest(const TString& imei, TInstant since, TInstant until) const {
    Y_ENSURE(TvmClient);
    NNeh::THttpRequest request;

    TStringBuilder cgiBuilder;
    cgiBuilder << "dbid=" << imei << "&uuid=" << imei;

    request.SetUri(Config.GetGetTrackPath())
        .SetRequestType("POST")
        .AddHeader("Content-Type", "application/json")
        .AddHeader("X-Ya-Service-Ticket", TvmClient->GetServiceTicketFor(Config.GetDestinationTvmId()))
        .SetCgiData(cgiBuilder);

    NJson::TJsonValue body;
    body["sources"].AppendValue("Verified");
    body["from_time"] = since.ToString();
    body["to_time"] = until.ToString();

    request.SetPostData(std::move(body));
    return request;
}

NNeh::THttpRequest TTaxiTrackStoryClient::CreateCameraRequest(const TString& uuid,const TString& dbid, TInstant since, TInstant until) const {
    Y_ENSURE(TvmClient);
    NNeh::THttpRequest request;

    TStringBuilder cgiBuilder;
    cgiBuilder << "dbid=" << dbid << "&uuid=" << uuid;

    request.SetUri(Config.GetGetCameraTrackPath())
        .SetRequestType("POST")
        .AddHeader("Content-Type", "application/json")
        .AddHeader("X-Ya-Service-Ticket", TvmClient->GetServiceTicketFor(Config.GetCameraDestinationTvmId()))
        .SetCgiData(cgiBuilder);

    NJson::TJsonValue body;
    body["from_time"] = since.ToString();
    body["to_time"] = until.ToString();

    request.SetPostData(std::move(body));
    return request;
}

NThreading::TFuture<NDrive::TTrack> TTaxiTrackStoryClient::GetTrack(const TString& imei, TInstant since, TInstant until) const {
    if (!TvmClient) {
        return NThreading::MakeErrorFuture<NDrive::TTrack>(std::make_exception_ptr(yexception() << "TvmClient not configured"));
    }
    auto request = CreateRequest(imei, since, until);
    auto response = Agent->SendAsync(request, Now() + Config.GetRequestTimeout());
    return response.Apply([](const NThreading::TFuture<NUtil::THttpReply>& r) {
        const auto& reply = r.GetValue();
        reply.EnsureSuccessfulReply();

        const auto track = NJson::ToJson(NJson::JsonString(reply.Content()));

        NDrive::TTrack result;

        if (track.GetArray().size() == 0) {
            return NThreading::MakeFuture(result);
        }

        if (!track.GetArray()[0].Has("Verified")) {
            return NThreading::MakeFuture(result);
        }

        for (auto& trackPoint : track.GetArray()[0]["Verified"].GetArray()) {
            if (trackPoint["geodata"].GetArray().size() == 0) {
                continue;
            }

            if (trackPoint["geodata"].GetArray()[0]["positions"].GetArray().size() == 0) {
                continue;
            }

            const auto& pos = trackPoint["geodata"].GetArray()[0]["positions"].GetArray()[0]["position"];
            const auto coord = TGeoCoord{pos[0].GetDouble(), pos[1].GetDouble()};
            const auto timestamp = TInstant::Seconds(trackPoint["timestamp"].GetInteger() / 1000);
            result.Coordinates.emplace_back(coord, timestamp);
        }
        return NThreading::MakeFuture(result);
    });
}



NThreading::TFuture<NDrive::TTrack> TTaxiTrackStoryClient::GetCameraTrack(const TString& uuid,const TString& dbid, TInstant since, TInstant until) const {
    if (!TvmClient) {
        return NThreading::MakeErrorFuture<NDrive::TTrack>(std::make_exception_ptr(yexception() << "TvmClient not configured"));
    }
    auto request = CreateCameraRequest(uuid, dbid, since, until);
    auto response = Agent->SendAsync(request, Now() + Config.GetRequestTimeout());
    return response.Apply([](const NThreading::TFuture<NUtil::THttpReply>& r) {
        const auto& reply = r.GetValue();
        reply.EnsureSuccessfulReply();

        const auto track = NJson::ToJson(NJson::JsonString(reply.Content()));
        NDrive::TTrack result;

        if (track.GetArray().size() == 0) {
            return NThreading::MakeFuture(result);
        }

        if (!track.GetArray()[0].Has("Verified")) {
            return NThreading::MakeFuture(result);
        }

        for (auto& trackPoint : track.GetArray()[0]["Verified"].GetArray()) {
            if (trackPoint["geodata"].GetArray().size() == 0) {
                continue;
            }

            if (trackPoint["geodata"].GetArray()[0]["positions"].GetArray().size() == 0) {
                continue;
            }

            const auto& pos = trackPoint["geodata"].GetArray()[0]["positions"].GetArray()[0]["position"];
            const auto coord = TGeoCoord{pos[0].GetDouble(), pos[1].GetDouble()};
            const auto timestamp = TInstant::Seconds(trackPoint["timestamp"].GetInteger() / 1000);
            double speed = 0;
            if (trackPoint["geodata"].GetArray()[0]["positions"].GetArray()[0].Has("speed")) {
                speed = trackPoint["geodata"].GetArray()[0]["positions"].GetArray()[0]["speed"].GetDouble();
            }
            result.Coordinates.emplace_back(coord, timestamp, speed);
        }
        return NThreading::MakeFuture(result);
    });
}

TDuration TTaxiTrackStoryClient::GetMaxInterval() const {
    return Config.GetMaxInterval();
}

TDuration TTaxiTrackStoryClient::GetCameraMaxInterval() const {
    return Config.GetCameraMaxInterval();
}
