#include "client.h"

#include <library/cpp/json/json_reader.h>

#include <util/charset/utf8.h>
#include <util/string/builder.h>

namespace {
    const auto CarSearchTemplate = NJson::ReadJsonFastTree(R"(
{
    "query": {
        "park": {
            "id": "PARK_ID",
            "driver_profile": {
                "id": [
                    "DRIVER_PROFILE_ID"
                ],
                "work_status": [
                    "working"
                ]
            }
        }
    },
    "fields": {
        "car": [
            "amenities",
            "booster_count",
            "brand",
            "callsign",
            "categories",
            "color",
            "mileage",
            "number",
            "status",
            "transmission",
            "year",
            "model"
        ]
    }
}
    )");

    const auto DriverSearchTemplate = NJson::ReadJsonFastTree(R"(
{
    "query": {
        "park": {
            "id": "PARK_ID",
            "driver_profile": {
                "work_status": [
                    "working"
                ]
            }
        }
    },
    "fields": {
        "account": [],
        "car": [
            "number"
        ],
        "current_status": [],
        "driver_profile": [
            "id",
            "phones",
            "last_name"
        ],
        "park": []
    }
}
    )");
}

DECLARE_FIELDS_JSON_SERIALIZER(NDrive::TTaxiFleetClient::TCar);

NDrive::TTaxiFleetClient::TTaxiFleetClient(const TOptions& options)
    : Client(options.Endpoint)
{
}

NThreading::TFuture<NDrive::TTaxiFleetClient::TCars> NDrive::TTaxiFleetClient::ListCars(const TRequest& request, const TString& searchText) const {
    auto r = CreateRequest(request);
    r.SetUri("/v1/parks/cars/list");
    auto post = CarSearchTemplate;
    post["query"]["park"]["id"] = request.ParkId;
    if (searchText) {
        post["query"]["text"] = searchText;
    }
    r.SetPostData(post.GetStringRobust());

    auto reply = Client.SendAsync(r, Now() + request.Timeout);
    return reply.Apply([](const NThreading::TFuture<NNeh::THttpReply>& r) {
        const auto& reply = r.GetValue();
        Y_ENSURE(reply.IsSuccessReply(), reply.Code() << ' ' << reply.ErrorMessage() << ' ' << reply.Content());
        auto parsed = NJson::ReadJsonFastTree(reply.Content());
        return NJson::FromJson<TCars>(parsed["cars"]);
    });
}

NThreading::TFuture<NDrive::TTaxiFleetClient::TCar> NDrive::TTaxiFleetClient::CreateCar(const TCar& car, const TString idempotencyToken, const TRequest& request) {
    auto r = CreateRequest(request);
    r.AddHeader("X-Real-Ip", "::1");
    r.AddHeader("X-Idempotency-Token", idempotencyToken);
    r.SetUri("/v1/parks/cars");
    r.SetCgiData("park_id=" + request.ParkId);
    r.SetPostData(NJson::ToJson(car).GetStringRobust());

    auto reply = Client.SendAsync(r, Now() + request.Timeout);
    return reply.Apply([](const NThreading::TFuture<NNeh::THttpReply>& r) {
        const auto& reply = r.GetValue();
        Y_ENSURE(reply.IsSuccessReply(), reply.Code() << ' ' << reply.ErrorMessage() << ' ' << reply.Content());
        auto parsed = NJson::ReadJsonFastTree(reply.Content());
        return NJson::FromJson<TCar>(parsed);
    });
}

NThreading::TFuture<NDrive::TTaxiFleetClient::TCar> NDrive::TTaxiFleetClient::UpdateCar(const TCar& car, const TRequest& request) {
    auto r = CreateRequest(request);
    r.AddHeader("X-Real-Ip", "::1");
    r.SetUri("/v1/parks/cars");
    r.SetCgiData(TStringBuilder() << "park_id=" << request.ParkId << "&id=" << car.Id);
    r.SetPostData(NJson::ToJson(car).GetStringRobust(), "PUT");

    auto reply = Client.SendAsync(r, Now() + request.Timeout);
    return reply.Apply([](const NThreading::TFuture<NNeh::THttpReply>& r) {
        const auto& reply = r.GetValue();
        Y_ENSURE(reply.IsSuccessReply(), reply.Code() << ' ' << reply.ErrorMessage() << ' ' << reply.Content());
        auto parsed = NJson::ReadJsonFastTree(reply.Content());
        return NJson::FromJson<TCar>(parsed);
    });
}

NThreading::TFuture<NDrive::TTaxiFleetClient::TOptionalCar> NDrive::TTaxiFleetClient::GetActiveCar(const TString& driverProfileId, const TRequest& request) const {
    auto r = CreateRequest(request);
    r.SetUri("/v1/parks/driver-profiles/list");
    auto post = DriverSearchTemplate;
    post["query"]["park"]["id"] = request.ParkId;
    post["query"]["park"]["driver_profile"]["id"] = NJson::ToJson({ driverProfileId });
    r.SetPostData(post.GetStringRobust());

    auto reply = Client.SendAsync(r, Now() + request.Timeout);
    return reply.Apply([driverProfileId](const NThreading::TFuture<NNeh::THttpReply>& r) -> TOptionalCar {
        const auto& reply = r.GetValue();
        Y_ENSURE(reply.IsSuccessReply(), reply.Code() << ' ' << reply.ErrorMessage() << ' ' << reply.Content());
        auto parsed = NJson::ReadJsonFastTree(reply.Content());
        for (auto&& profile : parsed["driver_profiles"].GetArraySafe()) {
            const auto& id = profile["driver_profile"]["id"].GetStringSafe();
            if (id == driverProfileId) {
                return NJson::FromJson<TOptionalCar>(profile["car"]);
            }
        }
        ythrow yexception() << "driver_profile " << driverProfileId << " is not found";
    });
}

NThreading::TFuture<NDrive::TTaxiFleetClient::TOptionalCar> NDrive::TTaxiFleetClient::GetCarByDriveId(const TString& id, const TRequest& request) const {
    auto cars = ListCars(request, id);
    return cars.Apply([id](const NThreading::TFuture<NDrive::TTaxiFleetClient::TCars>& cars) -> TOptionalCar {
        for (auto&& car : cars.GetValue()) {
            if (car.Callsign == id) {
                return car;
            }
        }
        return {};
    });
}

NThreading::TFuture<NDrive::TTaxiFleetClient::TOptionalCar> NDrive::TTaxiFleetClient::GetCarByNumber(const TString& number, const TRequest& request) const {
    auto normalized = ToUpperUTF8(number);
    auto cars = ListCars(request, normalized);
    return cars.Apply([normalized](const NThreading::TFuture<NDrive::TTaxiFleetClient::TCars>& cars) -> TOptionalCar {
        for (auto&& car : cars.GetValue()) {
            if (car.Number == normalized) {
                return car;
            }
        }
        return {};
    });
}

NThreading::TFuture<void> NDrive::TTaxiFleetClient::Link(const TString& carId, const TString& driverProfileId, const TRequest& request) {
    auto r = CreateRequest(request);
    r.SetUri("/v1/parks/driver-profiles/car-bindings");
    r.SetCgiData(TStringBuilder()
        << "&car_id=" << carId
        << "&driver_profile_id=" << driverProfileId
        << "&park_id=" << request.ParkId
    );
    r.SetRequestType("PUT");

    auto reply = Client.SendAsync(r, Now() + request.Timeout);
    return reply.Apply([driverProfileId](const NThreading::TFuture<NNeh::THttpReply>& r) {
        const auto& reply = r.GetValue();
        Y_ENSURE(reply.IsSuccessReply(), reply.Code() << ' ' << reply.ErrorMessage() << ' ' << reply.Content());
        Cerr << reply.Content() << Endl;
    });
}

NNeh::THttpRequest NDrive::TTaxiFleetClient::CreateRequest(const TRequest& request) const {
    NNeh::THttpRequest result;
    result.AddHeader("X-API-Key", request.Token);
    result.AddHeader("X-Client-ID", request.ClientId);
    return result;
}
