#include "client.h"

#include <library/cpp/json/json_reader.h>

#include <util/charset/utf8.h>
#include <util/string/builder.h>

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TTaxiFleetVehiclesClient::TRecognizedData& result) {
    return
        NJson::ParseField(value, "park_id", result.MutableParkId(), false) &&
        NJson::ParseField(value, "car_id", result.MutableCarId(), false) &&
        NJson::ParseField(value, "color", result.MutableColor(), false) &&
        NJson::ParseField(value, "number", result.MutableNumber(), false) &&
        NJson::ParseField(value, "number_normalized", result.MutableNumberNormalized(), false) &&
        NJson::ParseField(value, "vin", result.MutableVin(), false) &&
        NJson::ParseField(value, "callsign", result.MutableCallsign(), false) &&
        NJson::ParseField(value, "model", result.MutableModel(), false) &&
        NJson::ParseField(value, "brand", result.MutableBrand(), false) &&
        NJson::ParseField(value, "year", result.MutableYear(), false) &&
        NJson::ParseField(value, "transmission", result.MutableTransmission(), false) &&
        NJson::ParseField(value, "status", result.MutableStatus(), false) &&
        NJson::ParseField(value, "mileage", result.MutableMileage(), false) &&
        NJson::ParseField(value, "carrier_permit_owner_id", result.MutableCarrierPermitOwnerId(), false) &&
        NJson::ParseField(value, "body_number", result.MutableBodyNumber(), false) &&
        NJson::ParseField(value, "euro_car_segment", result.MutableEuroCarSegment(), false) &&
        NJson::ParseField(value, "registration_cert", result.MutableRegistrationCert(), false) &&
        NJson::ParseField(value, "cert_verification", result.MutableCertVerification(), false) &&
        NJson::ParseField(value, "description", result.MutableDescription(), false);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TTaxiFleetVehiclesClient::TRecognizedVehicle& result) {
    return
        NJson::ParseField(value, "park_id_car_id", result.MutableParkIdCarId(), true) &&
        NJson::ParseField(value, "revision", result.MutableRevision(), false) &&
        NJson::ParseField(value, "data", result.MutableData(), false);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TTaxiFleetVehiclesClient::TRecognizedVehicles& result) {
    return
        NJson::ParseField(value, "vehicles", result.MutableRecognizedVehicle(), true);
}

NJson::TJsonValue TTaxiFleetVehiclesClient::TRequestParams::ToJson() const {
    NJson::TJsonValue result = NJson::JSON_MAP;
    result.InsertValue("numbers_in_set", NJson::ToJson(GetNumbers()));
    if (GetProjection().size()) {
        result.InsertValue("projection", NJson::ToJson(GetProjection()));
    }
    return result;
}

NThreading::TFuture<TTaxiFleetVehiclesClient::TRecognizedVehicles> TTaxiFleetVehiclesClient::GetCarsByNumbers(const TRequestParams& requestParams, TAtomicSharedPtr<NDrive::TEventLog::TState> eventLogState) const {
    if (!Tvm) {
       throw yexception() << "TTaxiFleetVehiclesClient::GetCarsByNumbers TvmClient not configured";
    }
    auto req = CreateRequest();
    req.SetUri(Config.GetEndpoint());
    req.SetPostData(requestParams.ToJson().GetStringRobust());

    return Client->SendAsync(req, Now() + Config.GetRequestTimeout())
        .Apply([eventLogState = std::move(eventLogState)](const NThreading::TFuture<NNeh::THttpReply>& response) {
            const auto& reply = response.GetValue();
            if (eventLogState) {
                NDrive::TEventLog::TStateGuard stateGuard(*eventLogState);
                if (response.HasValue()) {
                    NDrive::TEventLog::Log("TTaxiFleetVehiclesClient", NJson::TMapBuilder
                        ("code", reply.Code())
                        ("response", reply.Content())
                    );
                }
            }
            reply.EnsureSuccessfulReply();
            NJson::TJsonValue replyJson;
            if (!NJson::ReadJsonFastTree(reply.Content(), &replyJson)) {
                throw yexception() << "Cannot parse reply json " << reply.Content();
            }
            TRecognizedVehicles recognizedVehicles;
            if (!TryFromJson(replyJson, recognizedVehicles)) {
                throw yexception() << "Cannot suggest data from reply";
            }
            return NThreading::MakeFuture(recognizedVehicles);
        }
    );
}

NNeh::THttpRequest TTaxiFleetVehiclesClient::CreateRequest() const {
    NNeh::THttpRequest result;
    result.AddHeader("Content-Type", "application/json")
        .AddHeader("X-Ya-Service-Ticket", Tvm->GetServiceTicketFor(Config.GetDestinationTvmId()));

    auto cgiData = result.GetCgiData();
    result.SetCgiData(
        TStringBuilder()
        << cgiData
        << ((cgiData) ? "&" : "")
        << "consumer=" << Config.GetConsumer()
    );
    return result;
}

TTaxiFleetVehiclesClient::TTaxiFleetVehiclesClient(const TTaxiFleetVehiclesConfig& config, TAtomicSharedPtr<NTvmAuth::TTvmClient> tvm)
    : Config(config)
    , Tvm(tvm)
{
    Client = MakeHolder<NNeh::THttpClient>(Config.GetUri(), Config.GetRequestConfig());
}
