#include "client.h"

#include <library/cpp/json/json_reader.h>

#include <rtline/util/algorithm/container.h>
#include <rtline/library/unistat/cache.h>
#include <rtline/library/json/builder.h>
#include <rtline/library/json/parse.h>

#include <util/generic/yexception.h>
#include <util/string/builder.h>
#include <util/system/env.h>

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::TTaxiPromocodesClient::TPromocode& result) {
    return NJson::ParseField(value["promocode"], result.Promocode);
}

namespace NDrive {
    TTaxiPromocodesClient::TTaxiPromocodesClient(const TOptions& options,  TMaybe<TTvmAuth> tvmAuth)
        : Client(MakeHolder<NNeh::THttpClient>(options.Endpoint))
        , Options(options)
        , TvmAuth(tvmAuth)
    {
    }

    NThreading::TFuture<TString> TTaxiPromocodesClient::GeneratePromocode(const TRequestData& requestData) const {
        NNeh::THttpRequest request;
        request.SetUri("/internal/generate");
        if (TvmAuth) {
            TvmAuth->UpdateRequest(request);
        }
        request.SetPostData(requestData.GetGenerateRequestData());

        TUnistatSignalsCache::SignalAdd("taxi_promocodes_client", "request", 1);
        TInstant start = Now();
        auto reply = Yensured(Client)->SendAsync(request, Now() + Options.Timeout);
        return reply.Apply([start](const NThreading::TFuture<NUtil::THttpReply>& r) -> TString {
            const TDuration duration = Now() - start;
            const NUtil::THttpReply& reply = r.GetValue();
            NJson::TJsonValue json;
            CheckReply(reply);
            TUnistatSignalsCache::SignalHistogram("taxi_promocodes_client", "response-times", duration.MilliSeconds(), NRTLineHistogramSignals::IntervalsRTLineReply);
            json = NJson::ReadJsonFastTree(reply.Content());
            if (!json.IsDefined() || !json.Has("promocode")) {
                TUnistatSignalsCache::SignalAdd("taxi_promocodes_client", "generate-promocode_error", 1);
                ythrow yexception() << "TTaxiPromocodesClient: response doesn't have field promocode. Error: " <<  reply.Serialize() << Endl;
            }
            TPromocode promo;
            if (!NJson::TryFromJson(json, promo)) {
                ythrow yexception() << "TTaxiPromocodesClient: cannot parse field promocode" << Endl;
            }
            return promo.Promocode;
        });
    }

    void TTaxiPromocodesClient::CheckReply(const NUtil::THttpReply& reply) {
        if (reply.IsUserError()) {
            TUnistatSignalsCache::SignalAdd("taxi_promocodes_client", "response-user_error", 1);
        }
        if (reply.IsServerError()) {
            TUnistatSignalsCache::SignalAdd("taxi_promocodes_client", "response-server_error", 1);
        }
        reply.EnsureSuccessfulReply();
        TUnistatSignalsCache::SignalAdd("taxi_promocodes_client", "response-success", 1);
    }

    NJson::TJsonValue TTaxiPromocodesClient::TRequestData::GetGenerateRequestData() const {
        NJson::TJsonValue data;
        data["token"] = Token;
        data["series_id"] = SeriesId;
        data["yandex_uid"] = YandexUid;
        data["brand_name"] = BrandName;
        data["expire_at"] = ExpireAt.FormatLocalTime("%Y-%m-%dT%H:%M:%S+00:00");
        return data;
    }

    TDuration TTaxiPromocodesClient::GetTimeout() const {
        return Options.Timeout;
    }
}
