#pragma once

#include "async.h"
#include "config.h"
#include "taxi_report.h"

#include <rtline/library/geometry/coord.h>
#include <rtline/library/async_proxy/async_delivery.h>

#include <util/generic/maybe.h>

class TTaxiRequest {
public:
    using TPtr = TAtomicSharedPtr<TTaxiRequest>;

    static const TString ECONOM_CLASS;
    static const TString COMFORT_CLASS;
    static const TString COMFORT_PLUS_CLASS;
    static const TString BUSINESS_CLASS;

    TTaxiRequest(const TGeoCoord& from, const TMaybe<TGeoCoord>& to, const TString& taxiClass = ECONOM_CLASS)
        : From(from)
        , To(to)
        , TaxiClass(taxiClass) {}

    TString Serialize(const TString& clientId) const {
        TString result = "?req=yellowcarnumber&class=" + TaxiClass + "&clid=" + clientId + "&rll=" + ::ToString(From.X) + "," + ::ToString(From.Y);
        if (To.Defined()) {
            result += "~" + ::ToString(To->X) + "," + ::ToString(To->Y);
        }
        if (Language) {
            result += "&lang=" + Language;
        }
        return result;
    }

    const TGeoCoord& GetTo() const {
        CHECK_WITH_LOG(To.Defined());
        return *To;
    }

    const TString& GetLanguage() const {
        return Language;
    }

    void SetLanguage(const TString& value) {
        Language = value;
    }

private:
    TGeoCoord From;
    TMaybe<TGeoCoord> To;
    TString Language = "en";
    TString TaxiClass;
};

class TSimpleTaxiReply: public NRTLine::TAsyncReplyBase {
public:
    using TPtr = TAtomicSharedPtr<TSimpleTaxiReply>;
    TSimpleTaxiReply(NNeh::THandleRef handle, TDuration timeout);
    virtual bool ParseRawReply(const TString& rawData) const override;
    const TTaxiPriceReport& GetReport() const;

private:
    mutable TTaxiPriceReport Report;
};

class TSimpleTaxiClient {
public:
    TSimpleTaxiClient(const TTaxiModuleConfig& config);
    TSimpleTaxiReply::TPtr SendRequest(const TTaxiRequest& request, TDuration timeout) const;

private:
    const TTaxiModuleConfig& Config;
    const TString RequestBase;
};
