#include "client.h"

#include <rtline/library/json/adapters.h>
#include <rtline/library/json/parse.h>

NDrive::TTaxiRouteHistoryClient::TTaxiRouteHistoryClient(const TOptions& options, TAtomicSharedPtr<NTvmAuth::TTvmClient> tvm, TAsyncDelivery::TPtr ad)
    : Options(options)
    , Client(MakeAtomicShared<NNeh::THttpClient>(options.Endpoint, options.MetaConfig, ad))
    , Tvm(tvm)
{
}

NThreading::TFuture<void> NDrive::TTaxiRouteHistoryClient::Add(const NDrive::TTaxiRouteHistoryOrder& order, TDuration timeout) const {
    NNeh::THttpRequest request;
    request.SetUri(Options.QueueHandle);
    if (Tvm) {
        request.AddHeader("X-Ya-Service-Ticket", Tvm->GetServiceTicketFor(GetDestinationClientId()));
    }
    NJson::TJsonValue post;
    post["queue_name"] = Options.QueueName;
    post["task_id"] = order.Id;
    post["args"] = NJson::JSON_ARRAY;
    post["kwargs"]["order"] = NJson::ToJson(NJson::Pass(order));
    post["eta"] = ToString(TInstant::Zero());
    request.SetPostData(post.GetStringRobust());

    auto deadline = Now() + (timeout ? timeout : Options.DefaultTimeout);
    auto reply = Client->SendAsync(request, deadline);
    auto result = reply.Apply([](const NThreading::TFuture<NNeh::THttpReply>& r) {
        const auto& reply = r.GetValue();
        reply.EnsureSuccessfulReply();
    });
    return result;
}

NTvmAuth::TTvmId NDrive::TTaxiRouteHistoryClient::GetDestinationClientId() const {
    if (Options.DestinationClientId) {
        return Options.DestinationClientId;
    }
    if (Options.Endpoint == TaxiRouteHistoryProductionEndpoint) {
        return TaxiRouteHistoryProductionClientId;
    }
    if (Options.Endpoint == TaxiRouteHistoryTestingEndpoint) {
        return TaxiRouteHistoryTestingClientId;
    }
    return 0;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::TTaxiRouteHistoryPoint& object) {
    NJson::TJsonValue result;
    result["position"] = NJson::ToJson(object.Position);
    result["type"] = ToString(object.Type);
    return result;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::TTaxiRouteHistoryOrder& object) {
    NJson::TJsonValue result;
    result["order_id"] = object.Id;
    result["created"] = ToString(object.Created);
    result["yandex_uid"] = object.YandexUid;
    result["points"] = NJson::ToJson(object.Points);
    return result;
}
