#include "client.h"

#include <rtline/library/unistat/cache.h>
#include <rtline/library/json/builder.h>
#include <rtline/library/json/parse.h>

namespace NDrive {

    TScootersSurge::TScootersSurge(const TOptions& options, TMaybe<TTvmAuth> tvmAuth)
        : Client(MakeHolder<NNeh::THttpClient>(options.Endpoint))
        , Options(options)
        , TvmAuth(tvmAuth)
    {
    }

    TScootersSurge::TScootersSurge(const TScootersSurge& other)
        : Client(MakeHolder<NNeh::THttpClient>(other.Options.Endpoint))
        , Options(other.Options)
        , TvmAuth(other.TvmAuth)
    {
    }

    TScootersSurge& TScootersSurge::operator=(const TScootersSurge& other) {
        Options = other.Options;
        return *this;
    }

    NThreading::TFuture<TScootersSurge::TResult> TScootersSurge::GetScootersSurge(double latitude, double longitude) const {
        NNeh::THttpRequest request;
        request.SetUri("/v1/get-surge");
        request.AddHeader("Content-Type", "application/json");
        request.SetRequestType("POST");
        {
            NJson::TJsonValue request_body;
            NJson::TJsonValue& pointJson = request_body.InsertValue("point", NJson::JSON_MAP);
            pointJson.InsertValue("lat", latitude);
            pointJson.InsertValue("lon", longitude);
            request.SetPostData(request_body.GetStringRobust());
        }
        if (TvmAuth) {
            TvmAuth->UpdateRequest(request);
        } else {
            WARNING_LOG << "no TvmAuth for scooters-surge request" << Endl;
        }
        TUnistatSignalsCache::SignalAdd("scooters_surge", "request", 1);
        TInstant start = Now();
        auto reply = Yensured(Client)->SendAsync(request, start + Options.RequestTimeout);
        return reply.Apply([start](const NThreading::TFuture<NNeh::THttpReply>& r) {
            const TDuration duration = Now() - start;
            const NUtil::THttpReply& reply = r.GetValue();
            if (reply.IsUserError()) {
                TUnistatSignalsCache::SignalAdd("scooters_surge", "response-user_error", 1);
            }
            if (reply.IsServerError()) {
                TUnistatSignalsCache::SignalAdd("scooters_surge", "response-server_error", 1);
            }
            reply.EnsureSuccessfulReply();
            TUnistatSignalsCache::SignalAdd("scooters_surge", "response-success", 1);
            TUnistatSignalsCache::SignalHistogram(
                "scooters_surge", "response-times", duration.MilliSeconds(),
                NRTLineHistogramSignals::IntervalsRTLineReply
            );

            const auto response = NJson::ToJson(NJson::JsonString(reply.Content()));
            return NJson::FromJson<TResult>(response);
        });
    }

}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::TScootersSurge::TResult& result) {
    return
        NJson::ParseField(value["value"], result.Value) &&
        NJson::ParseField(value["features"], result.Features);
}
