#include "client.h"
#include "definitions.h"

#include <rtline/library/json/parse.h>

namespace NSignalq = NDrive::NSignalq;

THolder<ITaxiSignalqDrivematicsApiClient> ITaxiSignalqDrivematicsApiClient::ConstructTaxiSignalqDrivematicsApiClient(const TTaxiSignalqDrivematicsApiConfig &config, TAtomicSharedPtr<NTvmAuth::TTvmClient> tvmClient) {
    if (config.GetClientType() == "fake") {
        return MakeHolder<TFakeTaxiSignalqDrivematicsApiClient>();
    } else if (tvmClient) {
        return MakeHolder<TTaxiSignalqDrivematicsApiClient>(config, tvmClient);
    }
    return nullptr;
}

NThreading::TFuture<NSignalq::TV1StatusesRetrieveResponse>
TTaxiSignalqDrivematicsApiClient::GetSignalqStatuses(const NSignalq::TV1StatusesRetrieveRequest& requestParams) const {
    if (!TvmClient) {
        throw yexception() << "TTaxiSignalqDrivematicsApiClient::GetSignalqStatuses TvmClient not configured";
    }
    NNeh::THttpRequest request;
    request.SetUri(Config.GetV1StatusesRetriveEndpoint())
        .SetRequestType("POST")
        .AddHeader("Content-Type", "application/json")
        .AddHeader("X-Ya-Service-Ticket", TvmClient->GetServiceTicketFor(Config.GetDestinationTvmId()))
        .SetPostData(NJson::ToJson(requestParams));
    return Client->SendAsync(request, Now() + Config.GetRequestTimeout())
        .Apply([](const NThreading::TFuture<NNeh::THttpReply>& response) {
            const auto& reply = response.GetValue();
            reply.EnsureSuccessfulReply();
            NJson::TJsonValue replyJson;
            if (!NJson::ReadJsonFastTree(reply.Content(), &replyJson)) {
                throw yexception() << "Cannot parse reply json " << reply.Content();
            }
            NSignalq::TV1StatusesRetrieveResponse statusesRetrieveResponse;
            if (!TryFromJson(replyJson, statusesRetrieveResponse)) {
                throw yexception() << "Cannot extract data from reply";
            }
            return NThreading::MakeFuture(std::move(statusesRetrieveResponse));
        }
    );
}

NThreading::TFuture<NSignalq::TV1EventsRetrieveResponse>
TTaxiSignalqDrivematicsApiClient::GetSignalqEvents(const NSignalq::TV1EventsRetrieveRequest& requestParams) const {
    if (!TvmClient) {
        throw yexception() << "TTaxiSignalqDrivematicsApiClient::GetSignalqEvents TvmClient not configured";
    }
    NNeh::THttpRequest request;
    request.SetUri(Config.GetV1EventsRetrieveEndpoint())
        .SetRequestType("POST")
        .AddHeader("Content-Type", "application/json")
        .AddHeader("X-Ya-Service-Ticket", TvmClient->GetServiceTicketFor(Config.GetDestinationTvmId()))
        .SetPostData(NJson::ToJson(requestParams));
    return Client->SendAsync(request, Now() + Config.GetRequestTimeout())
        .Apply([](const NThreading::TFuture<NNeh::THttpReply>& response){
            const auto& reply = response.GetValue();
            reply.EnsureSuccessfulReply();
            NJson::TJsonValue replyJson;
            if (!NJson::ReadJsonFastTree(reply.Content(), &replyJson)) {
                throw yexception() << "Cannot parse reply json " << reply.Content();
            }
            NSignalq::TV1EventsRetrieveResponse eventsRetrieveResponse;
            if (!TryFromJson(replyJson, eventsRetrieveResponse)) {
                throw yexception() << "Cannot extract data from reply";
            }
            return NThreading::MakeFuture(std::move(eventsRetrieveResponse));
        }
    );
}

NThreading::TFuture<NSignalq::TV1DeviceStatusHistoryIntervalsResponse>
TTaxiSignalqDrivematicsApiClient::GetSignalqStatusHistoryIntervals(const NSignalq::TV1DeviceStatusHistoryIntervalsRequest& requestParams) const {
    return GetSignalqStatusHistoryIntervalsJson(requestParams)
        .Apply([](const NThreading::TFuture<NJson::TJsonValue>& response) {
            const auto replyJson = response.GetValue();
            NSignalq::TV1DeviceStatusHistoryIntervalsResponse deviceStatusHistoryIntervalsResponse;
            if (!TryFromJson(replyJson, deviceStatusHistoryIntervalsResponse)) {
                throw yexception() << "Cannot extract data from reply";
            }
            return NThreading::MakeFuture(std::move(deviceStatusHistoryIntervalsResponse));
        }
    );
}

NThreading::TFuture<NJson::TJsonValue>
TTaxiSignalqDrivematicsApiClient::GetSignalqStatusHistoryIntervalsJson(const NSignalq::TV1DeviceStatusHistoryIntervalsRequest& requestParams) const {
    if (!TvmClient) {
        throw yexception() << "TTaxiSignalqDrivematicsApiClient::GetSignalqStatusHistoryIntervals TvmClient not configured";
    }
    NNeh::THttpRequest request;
    request.SetUri(Config.GetV1DeviceStatusHistoryIntervalsEndpoint())
        .SetRequestType("POST")
        .AddHeader("Content-Type", "application/json")
        .AddHeader("X-Ya-Service-Ticket", TvmClient->GetServiceTicketFor(Config.GetDestinationTvmId()))
        .SetPostData(NJson::ToJson(requestParams));

    return Client->SendAsync(request, Now() + Config.GetRequestTimeout())
        .Apply([](const NThreading::TFuture<NNeh::THttpReply>& response) {
            const auto& reply = response.GetValue();
            NJson::TJsonValue replyJson;
            if (!NJson::ReadJsonFastTree(reply.Content(), &replyJson)) {
                throw yexception() << "Cannot parse reply json " << reply.Content();
            }
            if (reply.IsUserError()) {
                NSignalq::TErrorResponse errorResponse;
                if (NJson::TryFromJson(replyJson, errorResponse)) {
                    throw NDrive::NSignalq::TBadRequestException() << "Bad request to signalq-drivematics-api. Code: " << errorResponse.GetCode() <<  ". Message: " << errorResponse.GetMessage();
                }
                throw yexception() << "Cannot parse user error reply json as NSignalq::TErrorResponse: " << reply.Content();
            }
            reply.EnsureSuccessfulReply();
            return NThreading::MakeFuture(std::move(replyJson));
        }
    );
}

NThreading::TFuture<NSignalq::TV1EventRaiseUrlsGenerateResponse>
TTaxiSignalqDrivematicsApiClient::GetSignalqEventRaiseUrls(const NSignalq::TV1EventRaiseUrlsGenerateRequest& requestParams) const{
    if (!TvmClient) {
        throw yexception() << "TTaxiSignalqDrivematicsApiClient::GetSignalqEventRaiseUrls TvmClient not configured";
    }
    NNeh::THttpRequest request;
    request.SetUri(Config.GetV1EventRaiseUrlsGenerateEndpoint())
        .SetRequestType("POST")
        .AddHeader("Content-Type", "application/json")
        .AddHeader("X-Ya-Service-Ticket", TvmClient->GetServiceTicketFor(Config.GetDestinationTvmId()))
        .SetPostData(NJson::ToJson(requestParams));
    return Client->SendAsync(request, Now() + Config.GetRequestTimeout())
        .Apply([](const NThreading::TFuture<NNeh::THttpReply>& response) {
            const auto& reply = response.GetValue();
            reply.EnsureSuccessfulReply();
            NJson::TJsonValue replyJson;
            if (!NJson::ReadJsonFastTree(reply.Content(), &replyJson)) {
                throw yexception() << "Cannot parse reply json " << reply.Content();
            }
            NSignalq::TV1EventRaiseUrlsGenerateResponse eventRaiseUrlsGenerateResponse;
            if (!TryFromJson(replyJson, eventRaiseUrlsGenerateResponse)) {
                throw yexception() << "Cannot extract data from reply";
            }
            return NThreading::MakeFuture(std::move(eventRaiseUrlsGenerateResponse));
        }
    );
}

NThreading::TFuture<NSignalq::TV1EventsMediaPresignedUrlsGenerateResponse>
TTaxiSignalqDrivematicsApiClient::GetEventsMediaPresignedUrls(const NSignalq::TV1EventsMediaPresignedUrlsGenerateRequest& requestParams) const {
    if (!TvmClient) {
        throw yexception() << "TTaxiSignalqDrivematicsApiClient::GetEventsMediaPresignedUrls TvmClient not configured";
    }
    NNeh::THttpRequest request;
    request.SetUri(Config.GetV1EventsMediaPresignedUrlsGenerateEndpoint())
        .SetRequestType("POST")
        .AddHeader("Content-Type", "application/json")
        .AddHeader("X-Ya-Service-Ticket", TvmClient->GetServiceTicketFor(Config.GetDestinationTvmId()))
        .SetPostData(NJson::ToJson(requestParams));
    return Client->SendAsync(request, Now() + Config.GetRequestTimeout())
        .Apply([](const NThreading::TFuture<NNeh::THttpReply>& response) {
            const auto& reply = response.GetValue();
            reply.EnsureSuccessfulReply();
            NJson::TJsonValue replyJson;
            if (!NJson::ReadJsonFastTree(reply.Content(), &replyJson)) {
                throw yexception() << "Cannot parse reply json " << reply.Content();
            }
            NSignalq::TV1EventsMediaPresignedUrlsGenerateResponse eventsMediaPresignedUrlsResponse;
            if (!TryFromJson(replyJson, eventsMediaPresignedUrlsResponse)) {
                throw yexception() << "Cannot extract data from reply";
            }
            return NThreading::MakeFuture(std::move(eventsMediaPresignedUrlsResponse));
        }
    );
}

NThreading::TFuture<void>
TTaxiSignalqDrivematicsApiClient::SetEventResolutionInFleet(const NSignalq::TV1EventResolutionRequest& requestParams) const {
    if (!TvmClient) {
        throw yexception() << "TTaxiSignalqDrivematicsApiClient::GetEventsMediaPresignedUrls TvmClient not configured";
    }
    NNeh::THttpRequest request;
    request.SetUri(Config.GetV1EventsResolutionsEndpoint())
        .SetRequestType("POST")
        .AddHeader("Content-Type", "application/json")
        .AddHeader("X-Ya-Service-Ticket", TvmClient->GetServiceTicketFor(Config.GetDestinationTvmId()))
        .SetPostData(NJson::ToJson(requestParams));
    return Client->SendAsync(request, Now() + Config.GetRequestTimeout())
        .Apply([](const NThreading::TFuture<NNeh::THttpReply>& response) {
            const auto& reply = response.GetValue();
            reply.EnsureSuccessfulReply();
            return NThreading::MakeFuture();
        }
    );
}

NThreading::TFuture<NSignalq::TV1DeviceParkGetResponse>
TTaxiSignalqDrivematicsApiClient::GetPark(const NSignalq::TV1DeviceParkGetRequest& requestParams) const{
    if (!TvmClient) {
        throw yexception() << "TTaxiSignalqDrivematicsApiClient::GetPark TvmClient not configured";
    }
    NNeh::THttpRequest request;
    request.SetUri(Config.GetV1DeviceGetParkEndpoint())
        .SetRequestType("GET")
        .AddHeader("Content-Type", "application/json")
        .AddHeader("X-Ya-Service-Ticket", TvmClient->GetServiceTicketFor(Config.GetDestinationTvmId()))
        .SetCgiData(
            TStringBuilder() << "&serial_number=" << requestParams.GetSerialNumber()
        );
    return Client->SendAsync(request, Now() + Config.GetRequestTimeout())
        .Apply([](const NThreading::TFuture<NNeh::THttpReply>& response) {
            const auto& reply = response.GetValue();
            reply.EnsureSuccessfulReply();
            NJson::TJsonValue replyJson;
            if (!NJson::ReadJsonFastTree(reply.Content(), &replyJson)) {
                throw yexception() << "Cannot parse reply json " << reply.Content();
            }
            NSignalq::TV1DeviceParkGetResponse getParkResponse;
            if (!TryFromJson(replyJson, getParkResponse)) {
                throw yexception() << "Cannot extract data from reply";
            }
            return NThreading::MakeFuture(std::move(getParkResponse));
        }
    );
}

NThreading::TFuture<NSignalq::TV1StatusesRetrieveResponse>
TFakeTaxiSignalqDrivematicsApiClient::GetSignalqStatuses(const NSignalq::TV1StatusesRetrieveRequest& requestParams) const {
    TVector<NSignalq::TV1StatusesRetrieveResponseItem> items;
    NJson::TJsonMap status ({
        {"status_at", "2022-03-10T15:02:57Z"},
        {"cpu_temperature", "1000"},
        {"disk_bytes_free_space", "1000"},
        {"disk_bytes_total_space", "1000"},
        {"root_bytes_free_space", "1000"},
        {"root_bytes_total_space", "1000"},
        {"ram_bytes_free_space", "1000"},
        {"ram_bytes_free_space", "1000"},
        {"software_version", "0.1.6-3.linux-yandex_2.X-mp_95b8b98b_202202241131_release-0.1.6-3_yandex"},
        {"gnss", NJson::TJsonMap({
            {"lat", "21.8"},
            {"lon", "32.2"},
            {"accuracy_m", "14.9611"}
        })}
    });

    NJson::TJsonMap lastLocationJson ({
        {"lat", "32.8333"},
        {"lon", "44.3333"},
        {"accuracy_m", "123.22"},
        {"speed_kmph", "53.12"},
        {"direction_deg", "12"},
        {"updated_at", "2022-03-10T14:00:00Z"}
    });
    NSignalq::TLastLocation lastLocation;
    NJson::TryFromJson(lastLocationJson, lastLocation);

    NSignalq::TStatus signalqStatus;
    NJson::TryFromJson(status, signalqStatus);
    items.reserve(requestParams.GetSerialNumbers().size());
    for (uint i = 0 ; i < requestParams.GetSerialNumbers().size(); ++i) {
        NSignalq::TV1StatusesRetrieveResponseItem responseItem;
        responseItem.SetStatus(signalqStatus);
        responseItem.SetSerialNumber(requestParams.GetSerialNumbers()[i]);
        responseItem.SetLastLocation(lastLocation);
        items.push_back(responseItem);
    }
    NSignalq::TV1StatusesRetrieveResponse response;
    response.SetItems(std::move(items));
    return NThreading::MakeFuture(std::move(response));
}

NThreading::TFuture<NSignalq::TV1EventsRetrieveResponse>
TFakeTaxiSignalqDrivematicsApiClient::GetSignalqEvents(const NSignalq::TV1EventsRetrieveRequest& requestParams) const {
    TVector<NSignalq::TV1EventsRetrieveResponseItem> items;
    NJson::TJsonMap event({
        {"id", "id-1"},
        {"at", "2022-03-10T15:02:57Z"},
        {"type", "type-1"},
        {"s3_video_path", "vpath"},
        {"s3_external_video_path", "evpath"},
        {"s3_photo_path", "ppath"},
        {"s3_external_photo_path", "eppath"},
        {"signalled", "false"},
        {"gnss", NJson::TJsonMap({
            {"lat", "21.8"},
            {"lon", "32.2"},
            {"accuracy_m", "14.9611"}
        })}
    });

    NSignalq::TEvent signalqEvent;
    NJson::TryFromJson(event, signalqEvent);
    items.reserve(requestParams.GetSerialNumbers().size());
    for (uint i = 0; i < requestParams.GetSerialNumbers().size(); ++i) {
        NSignalq::TV1EventsRetrieveResponseItem responseItem;
        responseItem.SetEvent(signalqEvent);
        responseItem.SetSerialNumber(requestParams.GetSerialNumbers()[i]);
        items.push_back(std::move(responseItem));
    }
    NSignalq::TV1EventsRetrieveResponse response;
    response.SetItems(std::move(items));
    return NThreading::MakeFuture(std::move(response));
}

NThreading::TFuture<NSignalq::TV1DeviceStatusHistoryIntervalsResponse>
TFakeTaxiSignalqDrivematicsApiClient::GetSignalqStatusHistoryIntervals(const NSignalq::TV1DeviceStatusHistoryIntervalsRequest& requestParams) const {
    return GetSignalqStatusHistoryIntervalsJson(requestParams)
        .Apply([](const NThreading::TFuture<NJson::TJsonValue>& response) {
            const auto replyJson = response.GetValue();
            NSignalq::TV1DeviceStatusHistoryIntervalsResponse deviceStatusHistoryIntervalsResponse;
            TryFromJson(replyJson, deviceStatusHistoryIntervalsResponse);
            return NThreading::MakeFuture(std::move(deviceStatusHistoryIntervalsResponse));
        }
    );
}

NThreading::TFuture<NJson::TJsonValue>
TFakeTaxiSignalqDrivematicsApiClient::GetSignalqStatusHistoryIntervalsJson(const NSignalq::TV1DeviceStatusHistoryIntervalsRequest& requestParams) const {
    const auto& optionalPeriod = requestParams.OptionalPeriod();
    if (optionalPeriod) {
        if (optionalPeriod->GetFrom() >= optionalPeriod->GetTo()) {
            return NThreading::MakeErrorFuture<NJson::TJsonValue>(std::make_exception_ptr(
                NDrive::NSignalq::TBadRequestException() << "Bad request to signalq-drivematics-api. Code: 400. Message: bad period"));
        }
    }
    NJson::TJsonValue responseJson = NJson::JSON_MAP;
    NJson::TJsonValue& intervalsJson =  responseJson.InsertValue("intervals", NJson::JSON_ARRAY);
    NJson::TJsonValue intervalJson;
    intervalJson["start_at"] = optionalPeriod ? ToString(optionalPeriod->GetFrom()) : ToString(TInstant::Days(300));
    intervalJson["end_at"] = optionalPeriod ? ToString(optionalPeriod->GetTo()) : ToString(TInstant::Days(301));
    intervalJson["status"] = "turned_on";
    intervalsJson.AppendValue(std::move(intervalJson));
    return NThreading::MakeFuture(std::move(responseJson));
}

NThreading::TFuture<NSignalq::TV1EventRaiseUrlsGenerateResponse>
TFakeTaxiSignalqDrivematicsApiClient::GetSignalqEventRaiseUrls(const NSignalq::TV1EventRaiseUrlsGenerateRequest& requestParams) const {
    static const TString url = "some_url_to_trigger_event";

    TVector<NSignalq::TV1EventRaiseUrlsGenerateItem> items;
    items.reserve(requestParams.GetUrlParameteres().size());
    for (const auto& urlParametersItem: requestParams.GetUrlParameteres()) {
        NSignalq::TV1EventRaiseUrlsGenerateItem item;
        item.SetSerialNumber(urlParametersItem.GetSerialNumber());
        item.SetUrl(url);
        item.SetUnixTimestamp(urlParametersItem.GetUnixTimestamp());
    }

    NSignalq::TV1EventRaiseUrlsGenerateResponse response;
    response.SetItems(std::move(items));
    return NThreading::MakeFuture(std::move(response));
}

NThreading::TFuture<NSignalq::TV1EventsMediaPresignedUrlsGenerateResponse>
TFakeTaxiSignalqDrivematicsApiClient::GetEventsMediaPresignedUrls(const NSignalq::TV1EventsMediaPresignedUrlsGenerateRequest& requestParams) const {
    NSignalq::TV1EventsMediaPresignedUrlsGenerateResponse response;

    for (const auto& event : requestParams.GetEvents()) {
        if (requestParams.HasOnlyUploaded()) {}

        NSignalq::TV1EventWithGeneratedUrls response_event;
        response_event.SetSerialNumberEventId(event.GetSerialNumberEventId());
        if (event.HasS3VideoPath()) {
            response_event.SetS3VideoPresignedUrl(CreateFakePresignedUrl(event.GetS3VideoPathRef(), requestParams.GetLinksExpiresAt()));
        }
        if (event.HasS3ExternalVideoPath()) {
            response_event.SetS3ExternalVideoPresignedUrl(CreateFakePresignedUrl(event.GetS3ExternalVideoPathRef(), requestParams.GetLinksExpiresAt()));
        }
        if (event.HasS3PhotoPath()) {
            response_event.SetS3PhotoPresignedUrl(CreateFakePresignedUrl(event.GetS3PhotoPathRef(), requestParams.GetLinksExpiresAt()));
        }
        if (event.HasS3ExternalPhotoPath()) {
            response_event.SetS3ExternalPhotoPresignedUrl(CreateFakePresignedUrl(event.GetS3ExternalPhotoPathRef(), requestParams.GetLinksExpiresAt()));
        }
        response.MutableEvents().push_back(std::move(response_event));
    }

    return NThreading::MakeFuture(response);
}

NThreading::TFuture<void>
TFakeTaxiSignalqDrivematicsApiClient::SetEventResolutionInFleet(const NSignalq::TV1EventResolutionRequest& /*requestParams*/) const {
    return NThreading::MakeFuture();
}

NThreading::TFuture<NSignalq::TV1DeviceParkGetResponse>
TFakeTaxiSignalqDrivematicsApiClient::GetPark(const NSignalq::TV1DeviceParkGetRequest& /*requestParams*/) const {
    NSignalq::TV1DeviceParkGetResponse response;
    response.SetParkId("nopark");
    return NThreading::MakeFuture(std::move(response));
}
