#include "definitions.h"

namespace NDrive::NSignalq {

    TV1StatusesRetrieveResponse::TSerialNumberToDataMap
    TV1StatusesRetrieveResponse::BuildSerialNumberToDataMap() const {
        TV1StatusesRetrieveResponse::TSerialNumberToDataMap result;
        for (const auto& item: Items) {
            if (!item.HasLastLocation() && !item.HasStatus()) {
                continue;
            }

            TV1StatusesRetrieveResponse::TSerialNumberData data;
            data.SetStatus(item.OptionalStatus());
            data.SetLastLocation(item.OptionalLastLocation());
            result[item.GetSerialNumber()] = std::move(data);
        }
        return result;
    }

    TV1EventsRetrieveResponse::TSerialNumberToDataMap
    TV1EventsRetrieveResponse::BuildSerialNumberToDataMap() const {
        TV1EventsRetrieveResponse::TSerialNumberToDataMap result;
        for (const auto& item : Items) {
            if (!item.HasEvent()) {
                continue;
            }

            TV1EventsRetrieveResponse::TSerialNumberData data;
            data.SetEvent(item.OptionalEvent());
            result[item.GetSerialNumber()] = std::move(data);
        }
        return result;
    }

    TV1EventsMediaPresignedUrlsGenerateResponse::TSerialNumberEventIdToEventPresignedUrlsMap
    TV1EventsMediaPresignedUrlsGenerateResponse::BuildSerialNumberEventIdToEventPresignedUrlsMap(const TInstant linksExpiresAt) const {
        TV1EventsMediaPresignedUrlsGenerateResponse::TSerialNumberEventIdToEventPresignedUrlsMap result;
        for (const auto& event: Events) {
            TV1EventsMediaPresignedUrlsGenerateResponse::TEventPresignedUrls eventPresignedUrls;
            eventPresignedUrls.OptionalS3VideoPresignedUrl() = event.OptionalS3VideoPresignedUrl();
            eventPresignedUrls.OptionalS3ExternalVideoPresignedUrl() = event.OptionalS3ExternalVideoPresignedUrl();
            eventPresignedUrls.OptionalS3PhotoPresignedUrl() = event.OptionalS3PhotoPresignedUrl();
            eventPresignedUrls.OptionalS3ExternalPhotoPresignedUrl() = event.OptionalS3ExternalPhotoPresignedUrl();
            eventPresignedUrls.SetExpiresAt(linksExpiresAt);

            result[event.GetSerialNumberEventId()] = std::move(eventPresignedUrls);
        }
        return result;
    }

    TV1EventRaiseUrlsGenerateResponse::TSerialNumberUnixTimestampToDataMap
    TV1EventRaiseUrlsGenerateResponse::BuildSerialNumberUnixTimestampToDataMap() const {
        TV1EventRaiseUrlsGenerateResponse::TSerialNumberUnixTimestampToDataMap result;
        for (const auto& item: Items) {
            result[item.GetSerialNumber() + "_" + ToString(item.GetUnixTimestamp())] = item.GetUrl();
        }
        return result;
    }

} // namespace NDrive::NSignalq

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::NSignalq::TLastLocation& result) {
    return
        NJson::ParseField(value, "lat", result.MutableLat(), true) &&
        NJson::ParseField(value, "lon", result.MutableLon(), true) &&
        NJson::ParseField(value, "accuracy_m", result.MutableAccuracyM(), true) &&
        NJson::ParseField(value, "speed_kmph", result.MutableSpeedKmph(), true) &&
        NJson::ParseField(value, "direction_deg", result.MutableDirectionDeg(), true) &&
        NJson::ParseField(value, "updated_at", result.MutableUpdateAt(), true);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::NSignalq::TGnss& result) {
    return
        NJson::ParseField(value, "lat", result.MutableLat(), false) &&
        NJson::ParseField(value, "lon", result.MutableLon(), false) &&
        NJson::ParseField(value, "accuracy_m", result.MutableAccuracyM(), false) &&
        NJson::ParseField(value, "speed_kmph", result.MutableSpeedKmph(), false) &&
        NJson::ParseField(value, "direction_deg", result.MutableDirectionDeg(), false);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::NSignalq::TGpsPosition& result) {
    return
        NJson::ParseField(value, "lat", result.MutableLat(), false) &&
        NJson::ParseField(value, "lon", result.MutableLon(), false);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::NSignalq::TUploadQueue& result) {
    return
        NJson::ParseField(value, "photo", result.MutablePhoto(), false) &&
        NJson::ParseField(value, "video", result.MutableVideo(), false) &&
        NJson::ParseField(value, "log", result.MutableLog(), false) &&
        NJson::ParseField(value, "event", result.MutableEvent(), false);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::NSignalq::TStatus& result) {
    return
        NJson::ParseField(value, "status_at", result.MutableStatusAt(), false) &&
        NJson::ParseField(value, "cpu_temperature", result.MutableCpuTemperature(), false) &&
        NJson::ParseField(value, "disk_bytes_free_space", result.MutableDiskBytesFreeSpace(), false) &&
        NJson::ParseField(value, "disk_bytes_total_space", result.MutableDiskBytesTotalSpace(), false) &&
        NJson::ParseField(value, "root_bytes_free_space", result.MutableRootBytesFreeSpace(), false) &&
        NJson::ParseField(value, "root_bytes_total_space", result.MutableRootBytesTotalSpace(), false) &&
        NJson::ParseField(value, "ram_bytes_free_space", result.MutableRamBytesFreeSpace(), false) &&
        NJson::ParseField(value, "software_version", result.MutableSoftwareVersion(), false) &&
        NJson::ParseField(value, "uptime_ms", result.MutableUptimeMs(), false) &&
        NJson::ParseField(value, "gnss", result.MutableGnss(), false) &&
        NJson::ParseField(value, "gps_position", result.MutableGpsPosition(), false) &&
        NJson::ParseField(value, "position_updated_at", result.MutablePositionUpdateAt(), false) &&
        NJson::ParseField(value, "sim_imsi", result.MutableSimImsi(), false) &&
        NJson::ParseField(value, "sim_iccid", result.MutableSimIccid(), false) &&
        NJson::ParseField(value, "sim_phone_number", result.MutableSimPhoneNumber(), false) &&
        NJson::ParseField(value, "upload_queue", result.MutableUploadQueue(), false) &&
        NJson::ParseField(value, "states", result.MutableStates(), false);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::NSignalq::TEvent& result) {
    return
        NJson::ParseField(value, "id", result.MutableId(), true) &&
        NJson::ParseField(value, "at", result.MutableAt(), true) &&
        NJson::ParseField(value, "type", result.MutableType(), true) &&
        NJson::ParseField(value, "gnss", result.MutableGnss(), false) &&
        NJson::ParseField(value, "s3_video_path", result.MutableS3VideoPath(), false) &&
        NJson::ParseField(value, "s3_external_video_path", result.MutableS3ExternalVideoPath(), false) &&
        NJson::ParseField(value, "s3_photo_path", result.MutableS3PhotoPath(), false) &&
        NJson::ParseField(value, "s3_external_photo_path", result.MutableS3ExternalPhotoPath(), false) &&
        NJson::ParseField(value, "signalled", result.MutableSignalled(), false);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::NSignalq::TV1SignalqSignalCreateItem& result) {
    return
        NJson::ParseField(value, "serial_number", result.MutableSerialNumber(), true) &&
        NJson::ParseField(value, "event", result.MutableEvent(), true);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::NSignalq::TV1SignalqSignalCreateRequestData& result) {
    return
        NJson::ParseField(value, "events", result.MutableEvents(), true);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::NSignalq::TV1StatusesRetrieveResponseItem& result) {
    return
        NJson::ParseField(value, "serial_number", result.MutableSerialNumber(), true) &&
        NJson::ParseField(value, "status", result.MutableStatus(), false) &&
        NJson::ParseField(value, "last_location", result.MutableLastLocation(), false);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::NSignalq::TV1StatusesRetrieveResponse& result) {
    return
        NJson::ParseField(value, "items", result.MutableItems(), true);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::NSignalq::TV1EventsRetrieveResponseItem& result) {
    return
        NJson::ParseField(value, "serial_number", result.MutableSerialNumber(), true) &&
        NJson::ParseField(value, "event", result.MutableEvent(), false);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::NSignalq::TV1EventsRetrieveResponse& result) {
    return
        NJson::ParseField(value, "items", result.MutableItems(), true);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::NSignalq::TV1EventRaiseUrlsGenerateItem& result) {
    return
        NJson::ParseField(value, "serial_number", result.MutableSerialNumber(), true) &&
        NJson::ParseField(value, "unix_timestamp", result.MutableUnixTimestamp(), true) &&
        NJson::ParseField(value, "url", result.MutableUrl(), true);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::NSignalq::TV1EventRaiseUrlsGenerateResponse& result) {
    return
        NJson::ParseField(value, "items", result.MutableItems(), true);
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NSignalq::TV1StatusesRetrieveRequest& object) {
    NJson::TJsonValue result = NJson::JSON_MAP;
    NJson::TJsonValue& serialNumbersJson =  result.InsertValue("serial_numbers", NJson::JSON_ARRAY);
    for (const auto& serialNumber : object.GetSerialNumbers()) {
        serialNumbersJson.AppendValue(NJson::ToJson(serialNumber));
    }
    return result;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NSignalq::TV1EventsRetrieveRequest& object) {
    NJson::TJsonValue result = NJson::JSON_MAP;
    NJson::TJsonValue& serialNumbersJson = result.InsertValue("serial_numbers", NJson::JSON_ARRAY);
    for (const auto& serialNumber : object.GetSerialNumbers()) {
        serialNumbersJson.AppendValue(NJson::ToJson(serialNumber));
    }
    return result;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NSignalq::TGnss& object) {
    NJson::TJsonValue result;
    NJson::InsertNonNull(result, "lat", object.OptionalLat());
    NJson::InsertNonNull(result, "lon", object.OptionalLon());
    NJson::InsertNonNull(result, "accuracy_m", object.OptionalAccuracyM());
    NJson::InsertNonNull(result, "speed_kmph", object.OptionalSpeedKmph());
    NJson::InsertNonNull(result, "direction_deg", object.OptionalDirectionDeg());
    return result;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NSignalq::TEvent& object) {
    NJson::TJsonValue result;
    NJson::InsertField(result, "id", object.GetId());
    NJson::InsertField(result, "at", object.GetAt());
    NJson::InsertField(result, "type", object.GetType());
    NJson::InsertNonNull(result, "gnss", object.OptionalGnss());
    NJson::InsertNonNull(result, "s3_video_path", object.OptionalS3VideoPath());
    NJson::InsertNonNull(result, "s3_external_video_path", object.OptionalS3ExternalVideoPath());
    NJson::InsertNonNull(result, "s3_photo_path", object.OptionalS3PhotoPath());
    NJson::InsertNonNull(result, "s3_external_photo_path", object.OptionalS3ExternalPhotoPath());
    NJson::InsertNonNull(result, "signalled", object.OptionalSignalled());
    return result;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NSignalq::TUrlParameteresItem& object) {
    NJson::TJsonValue result = NJson::JSON_MAP;
    NJson::InsertField(result, "serial_number", object.GetSerialNumber());
    NJson::InsertField(result, "unix_timestamp", object.GetUnixTimestamp());
    NJson::InsertNonNull(result, "event_type", object.OptionalEventType());
    return result;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NSignalq::TV1EventRaiseUrlsGenerateRequest& object) {
    NJson::TJsonValue result = NJson::JSON_MAP;
    NJson::TJsonValue& eventsJson = result.InsertValue("url_parameteres", NJson::JSON_ARRAY);
    for (const auto& urlParameter : object.GetUrlParameteres()) {
        eventsJson.AppendValue(NJson::ToJson(urlParameter));
    }
    return result;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NSignalq::TV1EventWithMedia& object) {
    NJson::TJsonValue result = NJson::JSON_MAP;
    NJson::InsertField(result, "serial_number_event_id", object.GetSerialNumberEventId());
    NJson::InsertNonNull(result, "s3_video_path", object.OptionalS3VideoPath());
    NJson::InsertNonNull(result, "s3_external_video_path", object.OptionalS3ExternalVideoPath());
    NJson::InsertNonNull(result, "s3_photo_path", object.OptionalS3PhotoPath());
    NJson::InsertNonNull(result, "s3_external_photo_path", object.OptionalS3ExternalPhotoPath());
    return result;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NSignalq::TV1EventsMediaPresignedUrlsGenerateRequest& object) {
    NJson::TJsonValue result = NJson::JSON_MAP;
    NJson::InsertField(result, "links_expires_at", object.GetLinksExpiresAt().ToIsoStringLocalUpToSeconds());
    NJson::InsertNonNull(result, "only_uploaded", object.OptionalOnlyUploaded());

    NJson::TJsonValue& eventsJson = result.InsertValue("events", NJson::JSON_ARRAY);
    for (const auto& event : object.GetEvents()) {
        eventsJson.AppendValue(NJson::ToJson(event));
    }
    return result;
}

template<>
NJson::TJsonValue NJson::ToJson(const NDrive::NSignalq::TV1EventResolutionRequest& object) {
    NJson::TJsonValue result = NJson::JSON_MAP;
    NJson::InsertField(result, "serial_number", object.GetSerialNumber());
    NJson::InsertField(result, "event_id", object.GetEventId());
    NJson::InsertField(result, "resolution", ToString(object.GetResolution()));
    return result;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::NSignalq::TV1EventWithGeneratedUrls& result) {
    return
        NJson::ParseField(value, "serial_number_event_id", result.MutableSerialNumberEventId(), true) &&
        NJson::ParseField(value, "s3_video_presigned_url", result.MutableS3VideoPresignedUrl(), false) &&
        NJson::ParseField(value, "s3_external_video_presigned_url", result.MutableS3ExternalVideoPresignedUrl(), false) &&
        NJson::ParseField(value, "s3_photo_presigned_url", result.MutableS3PhotoPresignedUrl(), false) &&
        NJson::ParseField(value, "s3_external_photo_presigned_url", result.MutableS3ExternalPhotoPresignedUrl(), false);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::NSignalq::TV1EventsMediaPresignedUrlsGenerateResponse& result) {
    return
        NJson::ParseField(value, "events", result.MutableEvents(), true);
}

NJson::TJsonValue
NDrive::NSignalq::TV1EventsMediaPresignedUrlsGenerateResponse::TEventPresignedUrls::BuildReportJson() const {
    NJson::TJsonValue result = NJson::JSON_MAP;
    if (HasS3VideoPresignedUrl()) {
        NJson::TJsonValue presignedUrlJson = NJson::JSON_MAP;
        NJson::InsertField(presignedUrlJson, "link", GetS3VideoPresignedUrlRef());
        NJson::InsertField(presignedUrlJson, "expires_at", NJson::Seconds(GetExpiresAt()));

        NJson::TJsonValue videoJson = NJson::JSON_MAP;
        NJson::InsertField(videoJson, "presigned_url", presignedUrlJson);

        NJson::InsertField(result, "video", videoJson);
    }
    if (HasS3ExternalVideoPresignedUrl()) {
        NJson::TJsonValue presignedUrlJson = NJson::JSON_MAP;
        NJson::InsertField(presignedUrlJson, "link", GetS3ExternalVideoPresignedUrlRef());
        NJson::InsertField(presignedUrlJson, "expires_at", NJson::Seconds(GetExpiresAt()));

        NJson::TJsonValue externalVideoJson = NJson::JSON_MAP;
        NJson::InsertField(externalVideoJson, "presigned_url", presignedUrlJson);

        NJson::InsertField(result, "external_video", externalVideoJson);
    }
    if (HasS3PhotoPresignedUrl()) {
        NJson::TJsonValue presignedUrlJson = NJson::JSON_MAP;
        NJson::InsertField(presignedUrlJson, "link", GetS3PhotoPresignedUrlRef());
        NJson::InsertField(presignedUrlJson, "expires_at", NJson::Seconds(GetExpiresAt()));

        NJson::TJsonValue photoJson = NJson::JSON_MAP;
        NJson::InsertField(photoJson, "presigned_url", presignedUrlJson);

        NJson::InsertField(result, "photo", photoJson);
    }
    if (HasS3ExternalPhotoPresignedUrl()) {
        NJson::TJsonValue presignedUrlJson = NJson::JSON_MAP;
        NJson::InsertField(presignedUrlJson, "link", GetS3ExternalPhotoPresignedUrlRef());
        NJson::InsertField(presignedUrlJson, "expires_at", NJson::Seconds(GetExpiresAt()));

        NJson::TJsonValue externalPhotoJson = NJson::JSON_MAP;
        NJson::InsertField(externalPhotoJson, "presigned_url", presignedUrlJson);

        NJson::InsertField(result, "external_photo", externalPhotoJson);
    }
    return result;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NSignalq::TPeriod& object) {
    NJson::TJsonValue result;
    NJson::InsertField(result, "from", object.GetFrom().ToIsoStringLocalUpToSeconds());
    NJson::InsertField(result, "to", object.GetTo().ToIsoStringLocalUpToSeconds());
    return result;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NSignalq::TV1DeviceStatusHistoryIntervalsRequest& object) {
    NJson::TJsonValue result = NJson::JSON_MAP;
    NJson::InsertField(result, "serial_number", object.GetSerialNumber());
    NJson::InsertNonNull(result, "period", object.OptionalPeriod());
    NJson::InsertNonNull(result, "cursor", object.OptionalCursor());
    NJson::InsertNonNull(result, "statuses_intervals_limit", object.OptionalStatusesIntervalsLimit());
    return result;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::NSignalq::TV1DeviceStatusHistoryInterval& result) {
    return
        NJson::ParseField(value, "status", result.MutableStatus(), true) &&
        NJson::ParseField(value, "start_at", result.MutableStartAt(), true) &&
        NJson::ParseField(value, "end_at", result.MutableEndAt(), true);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::NSignalq::TV1DeviceStatusHistoryIntervalsResponse& result) {
    return
        NJson::ParseField(value, "intervals", result.MutableIntervals(), true) &&
        NJson::ParseField(value, "cursor", result.MutableCursor(), false);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::NSignalq::TErrorResponse& result) {
    return
        NJson::ParseField(value, "code", result.MutableCode(), true) &&
        NJson::ParseField(value, "message", result.MutableMessage(), true);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::NSignalq::TV1DeviceParkGetResponse& result) {
    return  NJson::ParseField(value, "park_id", result.MutableParkId(), true);
}
