#include "client.h"

#include <rtline/library/json/adapters.h>
#include <rtline/library/json/parse.h>

NDrive::TTaxiSuggestClient::TTaxiSuggestClient(const TOptions& options, TAtomicSharedPtr<NTvmAuth::TTvmClient> tvm, TAsyncDelivery::TPtr ad)
    : Options(options)
    , Client(MakeAtomicShared<NNeh::THttpClient>(options.Endpoint, options.MetaConfig, ad))
    , Tvm(tvm)
{
}

NThreading::TFuture<NDrive::TTaxiSuggest> NDrive::TTaxiSuggestClient::GetZeroSuggest(double latitude, double longitude, TStringBuf uid, TStringBuf userTicket, ELanguage language, TDuration timeout) const {
    NNeh::THttpRequest request;
    request.SetUri(Options.ZeroSuggestHandle);
    if (language != LANG_UNK) {
        request.AddHeader("X-Request-Language", IsoNameByLanguage(language));
    }
    if (uid) {
        request.AddHeader("X-Yandex-UID", uid);
    }
    if (userTicket) {
        request.AddHeader("X-Ya-User-Ticket", userTicket);
    }
    if (Tvm) {
        request.AddHeader("X-Ya-Service-Ticket", Tvm->GetServiceTicketFor(Options.DestinationClientId));
    }
    if (Options.RequestApplication) {
        request.AddHeader("X-Request-Application", Options.RequestApplication);
    }
    if (Options.YaTaxiPassFlags) {
        request.AddHeader("X-YaTaxi-Pass-Flags", Options.YaTaxiPassFlags);
    }
    if (Options.YaTaxiPhoneId) {
        request.AddHeader("X-Yataxi-PhoneId", Options.YaTaxiPhoneId);
    }
    if (Options.YaTaxiUserId) {
        request.AddHeader("X-Yataxi-UserId", Options.YaTaxiUserId);
    }
    NJson::TJsonValue post;
    post["type"] = "b";
    post["position"] = NJson::ToJson(std::tie(longitude, latitude));
    post["state"] = NJson::JSON_MAP;
    request.SetPostData(post.GetStringRobust());

    auto deadline = Now() + (timeout ? timeout : Options.DefaultTimeout);
    auto reply = Client->SendAsync(request, deadline);
    auto result = reply.Apply([](const NThreading::TFuture<NNeh::THttpReply>& r) {
        const auto& reply = r.GetValue();
        Y_ENSURE(reply.IsSuccessReply(), reply.Code() << ' ' << reply.Content());
        const auto response = NJson::ToJson(NJson::JsonString(reply.Content()));
        const auto result = NJson::FromJson<TTaxiSuggest>(response);
        return result;
    });
    return result;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::TTaxiSuggest::TElement& object) {
    NJson::TJsonValue result;
    result["position"] = NJson::ToJson(std::tie(object.Longitude, object.Latitude));
    result["text"] = object.Text;
    result["title"] = object.Title;
    result["subtitle"] = object.Subtitle;
    return result;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::TTaxiSuggest::TElement& result) {
    return
        NJson::ParseField(value["position"], std::tie(result.Longitude, result.Latitude)) &&
        NJson::ParseField(value["text"], result.Text) &&
        NJson::ParseField(value["title"]["text"], result.Title) &&
        NJson::ParseField(value["subtitle"]["text"], result.Subtitle);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::TTaxiSuggest& result) {
    return
        NJson::ParseField(value["results"], result.Elements);
}
