#include "client.h"

#include <drive/library/cpp/threading/future.h>

#include <rtline/library/json/parse.h>

#include <util/string/builder.h>


template <>
NJson::TJsonValue NJson::ToJson(const TTaxiSupportChatSuggestClient::TDialog& dialog) {
    NJson::TJsonValue result;
    NJson::TJsonValue& dialogJson = result.InsertValue("dialog", NJson::JSON_MAP);
    NJson::InsertField(dialogJson, "messages", dialog.GetMessages());
    if (dialog.GetFeatures().IsDefined()) {
        NJson::InsertField(result, "features", dialog.GetFeatures());
    }
    return result;
}

template <>
NJson::TJsonValue NJson::ToJson(const TTaxiSupportChatSuggestClient::TTaxiMessage& message) {
    NJson::TJsonValue result;
    NJson::InsertField(result, "text", message.GetText());
    NJson::InsertField(result, "id", message.GetId());
    NJson::InsertField(result, "author", ToString(message.GetAuthor()));
    return result;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TTaxiSupportChatSuggestClient::TSuggestResponse& result) {
    if (value["features"].IsMap()) {
        if (!NJson::ParseField(value["features"], "most_probable_topic", result.MutableMostProbableTopic(), true) ||
            !NJson::ParseField(value["features"], "sure_topic", result.MutableSureTopic())) {
            return false;
        }
        for (auto&& probabilityJson : value["features"]["probabilities"].GetArray()) {
            NDrive::TSupportPrediction::TElement prediction;
            if (!NJson::ParseField(probabilityJson, "topic_name", prediction.Topic, true)
                || !NJson::ParseField(probabilityJson, "probability", prediction.Probability, true)) {
                    return false;
            }
            result.MutablePredictions().Elements.emplace_back(std::move(prediction));
        }
    } else {
        return false;
    }
    return true;
}

NNeh::THttpRequest TTaxiSupportChatSuggestClient::CreateRequest(const TTaxiSupportChatSuggestClient::TDialog& dialog) const {
    NNeh::THttpRequest request;
    TStringBuilder uriBuilder;
    uriBuilder << Config.GetSupportSuggestPath()
                << "?project_id="
                << Config.GetProjectId();

    request.SetUri(uriBuilder)
        .SetRequestType("POST")
        .AddHeader("Content-Type", "application/json")
        .AddHeader("X-Ya-Service-Ticket", TvmClient->GetServiceTicketFor(Config.GetDestinationTvmId()))
        .SetPostData(NJson::ToJson(dialog));
    NDrive::TEventLog::Log("TaxiSupportChatSuggestClientRequest", NJson::TMapBuilder
        ("request", request.Serialize())
        ("uri", uriBuilder)
    );
    return request;
}

NThreading::TFuture<TTaxiSupportChatSuggestClient::TSuggestResponse> TTaxiSupportChatSuggestClient::GetChatSuggest(const TDialog& dialog) const {
    if (!TvmClient) {
        return NThreading::MakeErrorFuture<TTaxiSupportChatSuggestClient::TSuggestResponse>(std::make_exception_ptr(yexception() << "TvmClient not configured"));
    }
    auto request = CreateRequest(dialog);
    auto chatId = dialog.GetChatId();
    return Agent->SendAsync(request, Now() + Config.GetRequestTimeout()).Apply([chatId](const NThreading::TFuture<NUtil::THttpReply>& r) {
        if (r.HasException()) {
            throw NThreading::GetException(r);
        }
        if (!r.HasValue()) {
            return NThreading::MakeErrorFuture<TSuggestResponse>(std::make_exception_ptr(yexception() << "No value in http reply future"));
        }
        const auto& reply = r.GetValue();
        if (!reply.IsSuccessReply()) {
            return NThreading::MakeErrorFuture<TSuggestResponse>(std::make_exception_ptr(yexception() << "Request error, reply code " << reply.Code() << ", error: " << reply.ErrorMessage() << ", content: " << reply.Content()));
        };
        NJson::TJsonValue replyJson;
        if (!NJson::ReadJsonFastTree(reply.Content(), &replyJson)) {
            return NThreading::MakeErrorFuture<TSuggestResponse>(std::make_exception_ptr(yexception() << "Cannot parse reply json " << reply.Content()));
        }
        NDrive::TEventLog::Log("TTaxiSupportChatSuggestClient", NJson::TMapBuilder
            ("chat_id", chatId)
            ("reply", replyJson)
        );
        TTaxiSupportChatSuggestClient::TSuggestResponse suggestReply;
        if (!TryFromJson(replyJson, suggestReply)) {
            return NThreading::MakeErrorFuture<TSuggestResponse>(std::make_exception_ptr(yexception() << "Cannot suggest data from reply"));
        }
        return NThreading::MakeFuture(suggestReply);
    });
}
