#pragma once

#include "config.h"

#include <drive/backend/chat/message.h>
#include <drive/backend/chat_robots/ifaces.h>

#include <drive/library/cpp/taxi/support_classifier/client.h>
#include <rtline/util/network/neh.h>
#include <rtline/util/types/expected.h>
#include <library/cpp/tvmauth/client/facade.h>

class TTaxiSupportChatSuggestClient {
public:
    enum class EMessageAuthor {
        Support /* "support" */,
        User /* "user" */,
        AI /* "ai" */,
    };

    class TTaxiMessage {
        R_FIELD(TString, Id);
        R_FIELD(TString, Text);
        R_FIELD(EMessageAuthor, Author);

    public:
        TTaxiMessage(const IChatUserContext::TPtr context, const NDrive::NChat::TMessageEvent& message) {
            Id = ToString(message.GetHistoryEventId());
            Text = message.GetText();
            Author = message.GetHistoryUserId() == context->GetUserId() ? EMessageAuthor::User : EMessageAuthor::Support;
        }
    };

    class TDialog {

        R_FIELD(TString, ChatId);
        R_FIELD(TVector<TTaxiSupportChatSuggestClient::TTaxiMessage>, Messages);
        R_FIELD(NJson::TJsonValue, Features);

    public:
        TDialog(const IChatUserContext::TPtr context, const TVector<NDrive::NChat::TMessageEvent>& messages, NJson::TJsonValue features = NJson::JSON_NULL, const size_t maxMessageCount = 15)
            : Features(features)
        {
            ChatId = context->GetChatId() + "." + context->GetChatTopic();
            size_t messagesCount = 0;
            for (auto&& message : Reversed(messages)) {
                if (!message.GetHistoryUserId().StartsWith("robot")) {
                    Messages.emplace_back(TTaxiSupportChatSuggestClient::TTaxiMessage(context, message));
                    ++messagesCount;
                }
                if (messagesCount >= maxMessageCount) {
                    break;
                }
            }
            std::reverse(Messages.begin(), Messages.end());
        }
    };

    class TSuggestResponse {
        R_FIELD(TString, SureTopic);
        R_FIELD(TString, MostProbableTopic);
        R_FIELD(NDrive::TSupportPrediction, Predictions);
    };

public:
    TTaxiSupportChatSuggestClient(const TTaxiSupportChatSuggestConfig& config, TAtomicSharedPtr<NTvmAuth::TTvmClient> tvmClient)
        : Config(config)
        , TvmClient(tvmClient)
    {
        Agent = MakeHolder<NNeh::THttpClient>(Config.GetUri(), Config.GetRequestConfig());
    }

    NThreading::TFuture<TSuggestResponse> GetChatSuggest(const TDialog& dialog) const;

private:
    NNeh::THttpRequest CreateRequest(const TTaxiSupportChatSuggestClient::TDialog& dialog) const;

    THolder<NNeh::THttpClient> Agent;
    const TTaxiSupportChatSuggestConfig& Config;
    TAtomicSharedPtr<NTvmAuth::TTvmClient> TvmClient;
};
