#include "client.h"

#include <drive/backend/logging/events.h>

#include <rtline/library/json/adapters.h>
#include <rtline/library/json/parse.h>


NDrive::TTaxiSupportClassifierClient::TTaxiSupportClassifierClient(const TOptions& options, TAtomicSharedPtr<NTvmAuth::TTvmClient> tvm, TAsyncDelivery::TPtr ad)
    : Options(options)
    , Client(MakeAtomicShared<NNeh::THttpClient>(options.IsTest ? options.EndPointTest : options.EndPointStable, options.MetaConfig, ad))
    , Tvm(tvm)
    , RegexId("^((image:)?\\w{4,}((-|:)\\w{4,})+)(,(image:)?\\w{4,}((-|:)\\w{4,})+)*$")
    , RegexTag("(\\[\\/?[a-zA-Z]*(?:=[^\\]]+)?\\])")
    , RegexUrl("http[s]?://(?:[a-zA-Z]|[0-9]|[#$-_@.&+]|[!*\\(\\),]|(?:%[0-9a-fA-F][0-9a-fA-F]))+")
{
}

NThreading::TFuture<NDrive::TSupportPrediction> NDrive::TTaxiSupportClassifierClient::Classify(const TSupportDialogue& dialogue, const TString& userId, const TString& mlRequestId, TDuration timeout)  const {
    NNeh::THttpRequest request;
    request.SetUri(Options.Uri);
    if (Tvm) {
        request.AddHeader("X-Ya-Service-Ticket", Tvm->GetServiceTicketFor(Options.IsTest ? Options.DestinationClientIdTest : Options.DestinationClientIdStable));
    }
    TSupportDialogue processedDialogue = Preprocess(dialogue);
    NJson::TJsonValue post;
    post["user_id"] = userId;
    post["ml_request_id"] = mlRequestId;
    post["dialogue"] = NJson::ToJson(processedDialogue);
    request.SetPostData(post.GetStringRobust());

    auto deadline = Now() + (timeout ? timeout : Options.DefaultTimeout);
    NDrive::TEventLog::Log("TaxiSupportClassifierRequest", NJson::TMapBuilder
        ("request", request.Serialize())
        ("deadline", deadline.ToString())
        ("user_id", userId)
        ("request_id", mlRequestId)
    );
    auto reply = Client->SendAsync(request, deadline);
    auto result = reply.Apply([userId, mlRequestId](const NThreading::TFuture<NNeh::THttpReply>& r) {
        const auto& reply = r.GetValue();
        NDrive::TEventLog::Log("TaxiSupportClassifierReply", NJson::TMapBuilder
            ("reply", reply.Serialize())
            ("user_id", userId)
            ("request_id", mlRequestId)
        );
        if (!reply.IsSuccessReply()) {
            throw NThreading::MakeErrorFuture<NDrive::TSupportPrediction>(std::make_exception_ptr(yexception() << reply.Code() << " " << reply.Content()));
        }
        const auto response = NJson::ToJson(NJson::JsonString(reply.Content()));
        const auto result = NJson::FromJson<TSupportPrediction>(response);
        return result;
    });
    return result;
}

TString NDrive::TTaxiSupportClassifierClient::CleanText(const TString& text) const {
    TString cleanedText = text;
    RE2::GlobalReplace(&cleanedText, RegexId, "");
    RE2::GlobalReplace(&cleanedText, RegexTag, "");
    RE2::GlobalReplace(&cleanedText, RegexUrl, "");
    cleanedText = StripInPlace(cleanedText);
    return cleanedText;
}

NDrive::TSupportDialogue NDrive::TTaxiSupportClassifierClient::Preprocess(const TSupportDialogue& dialogue) const {
    TSupportDialogue processedDialogue;
    for (auto&& dialogueTurn: dialogue) {
        TString authorId = dialogueTurn["author_id"].GetStringRobust();
        TString message = dialogueTurn["message"].GetStringRobust();
        TString cleanedMessage = CleanText(message);
        if (!authorId.StartsWith("robot") && cleanedMessage != "") {
            NJson::TJsonValue processedTurn;
            processedTurn["author_id"] = authorId;
            processedTurn["message"] = cleanedMessage;
            processedDialogue.push_back(processedTurn);
        }
    }
    return processedDialogue;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::TSupportPrediction::TElement& object) {
    NJson::TJsonValue result;
    result["topic"] = object.Topic;
    result["probability"] = object.Probability;
    return result;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::TSupportPrediction::TElement& result) {
    return
        NJson::ParseField(value["topic"], result.Topic) &&
        NJson::ParseField(value["probability"], result.Probability);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::TSupportPrediction& result) {
    return
        NJson::ParseField(value["top5_most_probable_topic"], result.Elements);
}
