#include "client.h"

#include <library/cpp/cgiparam/cgiparam.h>
#include <rtline/library/unistat/cache.h>
#include <rtline/library/json/builder.h>
#include <rtline/library/json/parse.h>

namespace NDrive {

    TTaxiSurgeCalculator::TTaxiSurgeCalculator(const TOptions& options, TMaybe<TTvmAuth> tvmAuth)
        : Client(new NNeh::THttpClient(options.Endpoint))
        , Options(options)
        , TvmAuth(tvmAuth)
    {
    }

    NThreading::TFuture<TTaxiSurgeCalculator::TResult> TTaxiSurgeCalculator::GetMapSurge(const TGeoCoord& location) const {
        TCgiParameters params;
        params.InsertUnescaped("point", Sprintf("%lf,%lf", location.X, location.Y));
        NNeh::THttpRequest request;
        request.SetUri("/v1/get-map-surge");
        request.SetCgiData(params.Print());
        if (TvmAuth) {
            TvmAuth->UpdateRequest(request);
        }
        TUnistatSignalsCache::SignalAdd("taxi_surge_calculator", "request", 1);
        TInstant start = Now();
        auto reply = Yensured(Client)->SendAsync(request, start + Options.RequestTimeout);
        return reply.Apply([start](const NThreading::TFuture<NNeh::THttpReply>& r) {
            const TDuration duration = Now() - start;
            const NUtil::THttpReply& reply = r.GetValue();
            if (reply.IsUserError()) {
                TUnistatSignalsCache::SignalAdd("taxi_surge_calculator", "response-user_error", 1);
            }
            if (reply.IsServerError()) {
                TUnistatSignalsCache::SignalAdd("taxi_surge_calculator", "response-server_error", 1);
            }
            reply.EnsureSuccessfulReply();
            TUnistatSignalsCache::SignalAdd("taxi_surge_calculator", "response-success", 1);
            TUnistatSignalsCache::SignalHistogram(
                "taxi_surge_calculator", "response-times", duration.MilliSeconds(),
                NRTLineHistogramSignals::IntervalsRTLineReply
            );
            const auto response = NJson::ToJson(NJson::JsonString(reply.Content()));
            return NJson::FromJson<TResult>(response);
        });
    }

}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::TTaxiSurgeCalculator::TResult& result) {
    return NJson::ParseField(value["classes"], result.Classes);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::TTaxiSurgeCalculator::TResult::TElement& result) {
    return
        NJson::ParseField(value["name"], result.Name) &&
        NJson::ParseField(value["value"], result.Value);
}
