#include "client.h"

#include <rtline/library/json/adapters.h>
#include <rtline/library/json/parse.h>

NDrive::TTaxiCommunicationsClient::TTaxiCommunicationsClient(const TOptions& options, TAtomicSharedPtr<NTvmAuth::TTvmClient> tvm, TAsyncDelivery::TPtr ad)
    : Options(options)
    , Client(MakeAtomicShared<NNeh::THttpClient>(options.Endpoint, options.MetaConfig, ad))
    , Tvm(tvm)
{
}

NThreading::TFuture<void> NDrive::TTaxiCommunicationsClient::Push(
    NJson::TJsonValue&& data,
    const TString& taxiUserId,
    const TString& intent,
    const TString& idempotencyToken,
    TDuration ttl,
    TDuration timeout
) const {
    NNeh::THttpRequest request;
    request.SetUri(Options.PushHandle);
    if (idempotencyToken) {
        request.AddHeader("X-Idempotency-Token", idempotencyToken);
    }
    if (Tvm) {
        request.AddHeader("X-Ya-Service-Ticket", Tvm->GetServiceTicketFor(GetDestinationClientId()));
    }

    NJson::TJsonValue post;
    post["data"] = std::move(data);
    post["user"] = taxiUserId;
    post["intent"] = intent;
    if (ttl) {
        post["ttl"] = ttl.Seconds();
    }
    request.SetPostData(post.GetStringRobust());

    auto deadline = Now() + (timeout ? timeout : Options.DefaultTimeout);
    auto reply = Client->SendAsync(request, deadline);
    auto result = reply.Apply([](const NThreading::TFuture<NNeh::THttpReply>& r) {
        const auto& reply = r.GetValue();
        reply.EnsureSuccessfulReply();
    });
    return result;
}

ui32 NDrive::TTaxiCommunicationsClient::GetDestinationClientId() const {
    if (Options.DestinationClientId) {
        return Options.DestinationClientId;
    }
    if (Options.Endpoint == TaxiCommunicationsProductionEndpoint) {
        return TaxiCommunicationsProductionTvmId;
    }
    if (Options.Endpoint == TaxiCommunicationsTestingEndpoint) {
        return TaxiCommunicationsTestingTvmId;
    }
    return 0;
}
