#include "data.h"

namespace {
    const NYT::TNode::TListType& GetList(const NYT::TNode& data, TStringBuf name) {
        if (data.HasKey(name)) {
            return data[name].AsList();
        } else {
            return Default<NYT::TNode::TListType>();
        }
    }
    const TString& GetString(const NYT::TNode& data, TStringBuf name) {
        if (data.HasKey(name)) {
            return data[name].AsString();
        } else {
            return Default<TString>();
        }
    }
}

const NYT::TNode* NDrive::GetPositionDataPtr(const NYT::TNode& data) {
    for (auto&& record : GetList(data, "records")) {
        if (GetString(record, "type") != "record") {
            continue;
        }
        for (auto&& subrecord : GetList(record, "subrecords")) {
            if (!subrecord.HasKey("type")) {
                continue;
            }
            const TString& type = subrecord["type"].AsString();
            if (type == "position_data") {
                return &subrecord;
            }
        }
    }
    return nullptr;
}

const NYT::TNode& NDrive::GetPositionData(const NYT::TNode& data) {
    auto ptr = GetPositionDataPtr(data);
    Y_ENSURE(ptr, "cannot find position_data");
    return *ptr;
}

void NDrive::GetCustomParameters(const NYT::TNode& data, TMap<TString, double>& sensors) {
    for (auto&& record : GetList(data, "records")) {
        if (GetString(record, "type") != "record") {
            continue;
        }
        for (auto&& subrecord : GetList(record, "subrecords")) {
            if (!subrecord.HasKey("type")) {
                continue;
            }
            const TString& type = subrecord["type"].AsString();
            if (type != "custom_parameters") {
                continue;
            }
            const auto& params = subrecord["params"];
            if (params.IsNull()) {
                continue;
            }

            for (auto&& item : params.AsMap()) {
                sensors.insert({ item.first, item.second.ConvertTo<double>() } );
            }
        }
    }
}

TMap<TString, double> NDrive::GetCustomParameters(const NYT::TNode& data) {
    TMap<TString, double> result;
    NDrive::GetCustomParameters(data, result);
    return result;
}

TMaybe<double> NDrive::GetCustomParameter(const NYT::TNode& data, const TStringBuf name) {
    for (auto&& record : GetList(data, "records")) {
        if (GetString(record, "type") != "record") {
            continue;
        }
        for (auto&& subrecord : GetList(record, "subrecords")) {
            if (!subrecord.HasKey("type")) {
                continue;
            }
            const TString& type = subrecord["type"].AsString();
            if (type != "custom_parameters") {
                continue;
            }
            const auto& params = subrecord["params"];
            if (params.IsNull()) {
                continue;
            }
            if (params.HasKey(name)) {
                return params[name].ConvertTo<double>();
            }
        }
    }
    return {};
}

double NDrive::GetLatitude(const NYT::TNode& data) {
    return GetPositionData(data)["lat"].ConvertTo<double>();
}

double NDrive::GetLongitude(const NYT::TNode& data) {
    return GetPositionData(data)["lon"].ConvertTo<double>();
}

double NDrive::GetCourse(const NYT::TNode& data) {
    return GetPositionData(data)["course"].ConvertTo<double>();
}

double NDrive::GetSpeed(const NYT::TNode& data) {
    return GetPositionData(data)["speed"].ConvertTo<double>();
}

TInstant NDrive::GetTimestamp(const NYT::TNode& data) {
    for (auto&& record : GetList(data, "records")) {
        if (GetString(record, "type") != "record") {
            continue;
        }
        auto seconds = record["timestamp"].ConvertTo<double>();
        return TInstant::Seconds(seconds);
    }
    ythrow yexception() << "cannot find timestamp";
}
