#include "tex_builder.h"

#include <rtline/util/json_processing.h>

#include <kernel/daemon/config/daemon_config.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/string_utils/base64/base64.h>

#include <util/generic/guid.h>
#include <util/stream/buffered.h>
#include <util/stream/file.h>
#include <util/stream/fwd.h>
#include <util/string/builder.h>
#include <util/string/cast.h>
#include <util/string/strip.h>
#include <util/string/subst.h>
#include <util/string/vector.h>
#include <util/system/shellcommand.h>

namespace {
    TVector<TString> SpecialChars = { "$", "%", "&", "#", "_", "{", "}" };
    constexpr TStringBuf CertificateFileName = "certificate.jpg";
}

IOutputStream& operator<<(IOutputStream& output, const NTexBuilder::TTableCell& builder) {
    output << builder.ToString();
    return output;
}

namespace NTexBuilder {
    TString TTextStyle::Quote(const TString& str) {
        TString newString = str;
        SubstGlobal(newString, "\\", " \\textbackslash ");
        SubstGlobal(newString, "~", " \\textasciitilde ");
        SubstGlobal(newString, "^", " \\textasciicircum ");
        for (const auto& c : SpecialChars) {
            SubstGlobal(newString, c, "\\" + c);
        }
        return newString;
    }

    bool TTexBuilderConfig::Init(const TYandexConfig::Section& section) {
        const TYandexConfig::Directives& directives = section.GetDirectives();
        TexBuilderFolder = directives.Value("TexBuilderFolder", TexBuilderFolder);
        ConvertTimeout = directives.Value("ConvertTimeout", ConvertTimeout);
        return true;
    }

    void TTexBuilderConfig::ToString(IOutputStream& os) const {
        os << "TexBuilderFolder: " << TexBuilderFolder << Endl;
        os << "ConvertTimeout: " << ConvertTimeout << Endl;
    }

    bool TTexBuilderConfig::DeserializeFromJson(const NJson::TJsonValue& json) {
        JREAD_STRING_OPT(json, "folder", TexBuilderFolder);
        JREAD_DURATION_OPT(json, "convert_timeout", ConvertTimeout);
        return true;
    }

    NJson::TJsonValue TTexBuilderConfig::SerializeToJson() const {
        NJson::TJsonValue result;
        JWRITE(result, "folder", TexBuilderFolder);
        TJsonProcessor::WriteDurationString(result, "convert_timeout", ConvertTimeout);
        return result;
    }

    TTexBuilderConfig TTexBuilderConfig::ParseFromString(const TString& configStr) {
        TTexBuilderConfig result;
        TAnyYandexConfig config;
        CHECK_WITH_LOG(config.ParseMemory(configStr.data()));
        CHECK_WITH_LOG(config.GetRootSection());
        result.Init(*config.GetRootSection());
        return result;
    }

    TTexBuilder::TTexBuilder(const TTexBuilderConfig& config)
        : Config(config)
        , TmpFolder(Config.GetTexBuilderFolder() + "/" + CreateGuidAsString())
    {
    }

    bool TTexBuilder::Init() {
        try {
            TFsPath(Config.GetTexBuilderFolder()).MkDir();
            TFsPath(TmpFolder).MkDir();
        } catch (const yexception& e) {
            ERROR_LOG << e.what() << Endl;
            return false;
        }
        TStringBuilder ss;
        ss << "\\documentclass[a4paper, 14pt]{extarticle}" << Endl;
        ss << "\\usepackage[utf8]{inputenc}" << Endl;
        ss << "\\usepackage[T2A]{fontenc}" << Endl;
        ss << "\\usepackage[russian, english]{babel}" << Endl;
        ss << "\\usepackage[left = 25mm, top = 20mm, right = 20mm, bottom = 20mm, nohead, nofoot, heightrounded]{geometry}" << Endl;
        ss << "\\usepackage{tabularx}" << Endl;
        ss << "\\usepackage{graphicx}" << Endl;
        ss << "\\usepackage{parskip}" << Endl;
        ss << "\\usepackage{pdfpages}" << Endl;
        ss << "\\usepackage{amsfonts}" << Endl;
        Header = ss;
        return true;
    }

    TTexBuilder::~TTexBuilder() {
        try {
            TFsPath(TmpFolder).ForceDelete();
        } catch (const std::exception& e) {
            ERROR_LOG << "cannot destruct TexBuilder: " << FormatExc(e) << Endl;
        }
    }


    bool TTexBuilder::AddText(const TString& content, TMessagesCollector& /*messages*/) {
        Blocks.push_back(content);
        return true;
    }

    bool TTexBuilder::AddImage(const TString& content, TMessagesCollector& messages, EPosition position, const TString& caption) {
        TString filePath = TmpFolder + "/image-" + CreateGuidAsString() +".jpg";
        try {
            TFileOutput fileOutput(filePath);
            fileOutput << content;
        } catch (const yexception& e) {
            messages.AddMessage("file_output", e.what());
            return false;
        }
        TStringBuilder ss;
        ss << "\\begin{figure}[" << position << "!]" << Endl;
        ss << "\\includegraphics[width = \\linewidth]{" << filePath << "}" << Endl;
        if (!caption.empty()) {
            ss << "\\caption{" << caption << "}" << Endl;
        }
        ss << "\\end{figure}" << Endl;
        return AddText(ss, messages);
    }

    bool TTexBuilder::AddPDF(const TString& content, TMessagesCollector& messages) {
        TString filePath = TmpFolder + "/pdf-" + CreateGuidAsString() + ".pdf";
        try {
            TFileOutput fileOutput(filePath);
            fileOutput << content;
        } catch (const yexception& e) {
            messages.AddMessage("file_output", e.what());
            return false;
        }
        TStringBuilder ss;
        ss << "\\includepdf[pages=-]{" << filePath << "}" << Endl;
        return AddText(ss, messages);
    }

    TString TTexBuilder::GetDocument() const {
        TStringStream ss;
        ss << Header << Endl;

        if (Signature) {
            ss << "\\newcommand{\\certificate}[3] {" << Endl;
            ss << "    \\makebox[0pt][l]{" << Endl;
            ss << "            \\hspace*{#1}\\raisebox{-\\totalheight}[0pt][0pt]{" << Endl;
            ss << "        \\includegraphics[width={#2},height={#3}]{" << TmpFolder << "/" << CertificateFileName << "}}" << Endl;
            ss << "    }" << Endl;
            ss << "}" << Endl;
        } else {
            ss << "\\newcommand{\\certificate}[3] {" << Endl;
            ss << "   \\hspace*{0cm}" << Endl;
            ss << "}" << Endl;
        }

        for (const auto& var : Variables) {
            ss << "\\newcommand{\\" << var.first << "}{" << TTextStyle::Quote(var.second) << "}" << Endl;
        }

        ss << "\\begin{document}" << Endl;
        ss << "\\sloppy" << Endl;
        for (const auto& part : Blocks) {
            ss << part << Endl;
        }

        ss << "\\end{document}" << Endl;
        return ss.Str();
    }

    bool TTexBuilder::AddTable(const TVector<TVector<TTableCell>>& cells, TMessagesCollector& messages) {
        if (cells.empty()) {
            messages.AddMessage(__LOCATION__, "Empty table");
            return false;
        }

        ui32 rowSize = cells[0].size();
        for (const auto& row : cells) {
            if (row.size() != rowSize) {
                messages.AddMessage(__LOCATION__, "Inconsistent table");
                return false;
            }
        }

        TStringBuilder ss;
        ss << "\\begin{tabularx}{\\linewidth}{|";
        for (size_t i = 0; i < rowSize; ++i) {
            ss << "X|";
        }
        ss << "}" << Endl;
        ss << "\\hline" << Endl;

        for (const auto& row : cells) {
            ss << JoinVectorIntoString(row, " & ") << " \\\\" << Endl;
            ss << "\\hline" << Endl;
        }
        ss << R"(\end{tabularx} \\)" << Endl;

        return AddText(ss, messages);
    }


    bool TTexBuilder::AddVariable(const TString& name, const TString& value) {
        return Variables.insert({ name, value }).second;
    }

    bool TTexBuilder::AddVariables(const TMap<TString, TString>& variables) {
        for (const auto& var : variables) {
            if (!AddVariable(var.first, var.second)) {
                ERROR_LOG << var.first << " already exists" << Endl;
                return false;
            }
        }
        return true;
    }

    TBlob TTexBuilder::BuildFinalDocument(TMessagesCollector& messages) const {
        TString texDocument = GetDocument();
        TString texDocumentName = TmpFolder + "/tex_document-" + ToString(TInstant::Now().MilliSeconds());
        TString texDocumentNameFull = texDocumentName + ".tex";
        try {
            if (Signature) {
                TFileOutput fileOutput(TmpFolder + "/" + CertificateFileName);
                fileOutput << Base64Decode(Signature);
            }
            TFileOutput fileOutput(texDocumentNameFull);
            fileOutput << texDocument;
        } catch (const yexception& e) {
            messages.AddMessage("file_output", e.what());
            return TBlob();
        }

        try {
            TShellCommandOptions options;
            options.SetClearSignalMask(true);
            TShellCommand cmd("timeout " + ToString(Config.GetConvertTimeout().Seconds()) + " pdflatex -output-directory " + TmpFolder + " " + texDocumentNameFull, options);
            cmd.Run();
            cmd.Wait();
            if (cmd.GetStatus() == TShellCommand::SHELL_INTERNAL_ERROR) {
                messages.AddMessage(__LOCATION__, TStringBuilder() << "Can't convert tex to pdf: " << cmd.GetInternalError());
            } else if (cmd.GetStatus() == TShellCommand::SHELL_ERROR) {
                messages.AddMessage(__LOCATION__, TStringBuilder() << "Can't convert tex to pdf: " << cmd.GetError());
            } else if (cmd.GetStatus() == TShellCommand::SHELL_FINISHED) {
                if (TFsPath(texDocumentName + ".pdf").Exists()) {
                    return TBlob::FromString(Strip(TFileInput(texDocumentName + ".pdf").ReadAll()));
                } else {
                    messages.AddMessage(__LOCATION__, "Can't convert tex to pdf: pdf file doesn't exist");
                }
            } else {
                messages.AddMessage(__LOCATION__, TStringBuilder() << "Can't convert tex to pdf: incorrect status: " << (ui32)cmd.GetStatus());
            }
        } catch (const std::exception& e) {
            messages.AddMessage(__LOCATION__, TStringBuilder() << "Can't convert tex to pdf: error: " << FormatExc(e));
        }
        return TBlob();
    }
}
