#pragma once

#include <rtline/util/types/messages_collector.h>
#include <rtline/util/types/accessor.h>

#include <library/cpp/yconf/conf.h>

#include <util/datetime/base.h>
#include <util/memory/blob.h>

namespace NTexBuilder {
    enum ETextStyle {
        BoldFace /* "textbf" */,
        Italic /* "textit" */,
    };

    enum EPosition {
        Top /* "t" */,
        Bottom /* "b" */,
        Page /* "p" */,
        Here /* "h" */,
        RightHere /* "H" */,
    };

    class TTextStyle {
    public:
        TTextStyle(ETextStyle name)
            : Name(name)
        {}

        TString Apply(const TString& str) const {
            return "\\" + ToString(Name) + "{" + str + "}";
        }

        static TString Quote(const TString& str);
    private:
        ETextStyle Name;
    };

    class TTableCell {
    public:
        TTableCell(const TString& text)
            : Content(text) {}

        void AddStyle(ETextStyle style) {
            Styles.push_back(TTextStyle(style));
        }

        TString ToString() const {
            TString result = TTextStyle::Quote(Content);
            for (const auto& style : Styles) {
                result = style.Apply(result);
            }
            return result;
        }

    private:
        TString Content;
        TVector<TTextStyle> Styles;
    };

    class TTexBuilderConfig {
        R_READONLY(TString, TexBuilderFolder, ".");
        R_READONLY(TDuration, ConvertTimeout, TDuration::Seconds(5));

    public:
        static TTexBuilderConfig ParseFromString(const TString& configStr);
        bool Init(const TYandexConfig::Section& section);
        void ToString(IOutputStream& os) const;
        bool DeserializeFromJson(const NJson::TJsonValue& json);
        NJson::TJsonValue SerializeToJson() const;
    };

    class TTexBuilder {
        R_FIELD(TString, Header);
        R_FIELD(TString, Signature);

    public:
        TTexBuilder(const TTexBuilderConfig& config);
        ~TTexBuilder();
        bool Init();

        TBlob BuildFinalDocument(TMessagesCollector& errors) const;

        bool AddText(const TString& content, TMessagesCollector& messages);
        bool AddImage(const TString& content, TMessagesCollector& messages, EPosition location = EPosition::Page, const TString& caption = "");
        bool AddPDF(const TString& content, TMessagesCollector& messages);
        bool AddTable(const TVector<TVector<TTableCell>>& cells, TMessagesCollector& messages);
        bool AddVariable(const TString& name, const TString& value);
        bool AddVariables(const TMap<TString, TString>& map);
        void SetSignature(const TString& signature);

    private:
        TString GetDocument() const;
    private:
        const TTexBuilderConfig Config;
        TMap<TString, TString> Variables;
        TVector<TString> Blocks;
        TString TmpFolder;
    };
}
