#include <drive/library/cpp/tex_builder/tex_builder.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/logger/global/global.h>
#include <util/stream/file.h>
#include <util/string/strip.h>
#include <util/string/builder.h>

using namespace NTexBuilder;

Y_UNIT_TEST_SUITE(TexBuilder) {

    TTexBuilderConfig GetDefaultConfig() {
        return TTexBuilderConfig::ParseFromString("");
    }

    Y_UNIT_TEST(EmptyDocument) {
        InitGlobalLog2Console();

        TTexBuilder builder(GetDefaultConfig());
        CHECK_WITH_LOG(builder.Init());
        TMessagesCollector errors;
        CHECK_WITH_LOG(builder.BuildFinalDocument(errors).Empty());
    }

    Y_UNIT_TEST(SimpleDocument) {
        InitGlobalLog2Console();

        TTexBuilder builder(GetDefaultConfig());
        CHECK_WITH_LOG(builder.Init());
        TMessagesCollector errors;
        builder.AddText("text", errors);

        TBlob pdf = builder.BuildFinalDocument(errors);
        CHECK_WITH_LOG(!pdf.Empty()) << errors.GetStringReport();
        {
            TFsPath("./result-simple").MkDir();
            TFileOutput fileOutput("./result-simple/" + ToString(TInstant::Now().MicroSeconds()) + ".pdf");
            fileOutput << TStringBuf(pdf.AsCharPtr(), pdf.Size());
        }
    }

    Y_UNIT_TEST(AddImage) {
        InitGlobalLog2Console();

        TString image = Strip(TIFStream("test-drive-logo.png").ReadAll());
        TTexBuilder builder(GetDefaultConfig());
        CHECK_WITH_LOG(builder.Init());
        TMessagesCollector errors;
        CHECK_WITH_LOG(builder.AddImage(image, errors) == 1);
        TBlob pdf = builder.BuildFinalDocument(errors);
        CHECK_WITH_LOG(!pdf.Empty()) << errors.GetStringReport();
        {
            TFsPath("./result-image").MkDir();
            TFileOutput fileOutput("./result-image/" + ToString(TInstant::Now().MicroSeconds()) + ".pdf");
            fileOutput << TStringBuf(pdf.AsCharPtr(), pdf.Size());
        }
    }

    Y_UNIT_TEST(AddTable) {
        InitGlobalLog2Console();
        TTexBuilder builder(GetDefaultConfig());
        CHECK_WITH_LOG(builder.Init());

        TVector<TVector<TTableCell>> cells;
        TVector<TTableCell> keyRow = { {"key"}, {"value"} };
        for (auto& cell : keyRow) {
            cell.AddStyle(ETextStyle::BoldFace);
        }
        cells.push_back(keyRow);

        TVector<TTableCell> row = { {"keyValue"}, {"valueValue"} };
        for (size_t i = 0; i < 10; ++i) {
            cells.push_back(row);
        }
        TMessagesCollector errors;
        CHECK_WITH_LOG(builder.AddTable(cells, errors)) << errors.GetStringReport();
        CHECK_WITH_LOG(builder.AddTable(cells, errors)) << errors.GetStringReport();

        TBlob pdf = builder.BuildFinalDocument(errors);
        CHECK_WITH_LOG(!pdf.Empty()) << errors.GetStringReport();
        {
            TFsPath("./result-table").MkDir();
            TFileOutput fileOutput("./result-table/" + ToString(TInstant::Now().MicroSeconds()) + ".pdf");
            fileOutput << TStringBuf(pdf.AsCharPtr(), pdf.Size());
        }

        row.push_back({ "newValue" });
        cells.push_back(row);
        CHECK_WITH_LOG(!builder.AddTable(cells, errors));
    }

    Y_UNIT_TEST(SpecialChars) {
        InitGlobalLog2Console();
        TTexBuilder builder(GetDefaultConfig());
        CHECK_WITH_LOG(builder.Init());

        TString specialChars = "$&%#_{}`^~+=|\\<>";
        TMessagesCollector errors;
        builder.AddText(TTextStyle::Quote(specialChars), errors);
        TBlob pdf = builder.BuildFinalDocument(errors);
        CHECK_WITH_LOG(!pdf.Empty()) << errors.GetStringReport();
        {
            TFsPath("./result-special").MkDir();
            TFileOutput fileOutput("./result-special/" + ToString(TInstant::Now().MicroSeconds()) + ".pdf");
            fileOutput << TStringBuf(pdf.AsCharPtr(), pdf.Size());
        }
    }

    Y_UNIT_TEST(Variable) {
        InitGlobalLog2Console();
        TTexBuilder builder(GetDefaultConfig());
        CHECK_WITH_LOG(builder.Init());

        TMessagesCollector errors;
        builder.AddText("\\testCommand", errors);
        builder.AddVariables({ {"testCommand", "testValue"} });

        TBlob pdf = builder.BuildFinalDocument(errors);
        CHECK_WITH_LOG(!pdf.Empty()) << errors.GetStringReport();
        {
            TFsPath("./result-variable").MkDir();
            TFileOutput fileOutput("./result-variable/" + ToString(TInstant::Now().MicroSeconds()) + ".pdf");
            fileOutput << TStringBuf(pdf.AsCharPtr(), pdf.Size());
        }
        builder.AddText("\\testCommandNew", errors);
        CHECK_WITH_LOG(builder.BuildFinalDocument(errors).Empty());
    }
}
