#pragma once

#include "concurrent_common.h"

#include <library/cpp/cache/cache.h>

#include <util/generic/maybe.h>
#include <util/generic/ptr.h>

namespace NUtil {
    template <
        class TKey,
        class TValue,
        size_t ShardsCount = 64,
        class TCacheImpl = TLRUCache<TKey, TValue>,
        class TLocker = TAdaptiveLocker
    >
    class TConcurrentCache: public TConcurrentSharded<TKey, THolder<TCacheImpl>, ShardsCount, TLocker> {
    private:
        using TBase = TConcurrentSharded<TKey, THolder<TCacheImpl>, ShardsCount, TLocker>;
        using TShard = typename TBase::TShard;
        using TSharedGuard = typename TBase::TSharedGuard;
        using TUniqueGuard = typename TBase::TUniqueGuard;

    public:
        template <class... TArgs>
        TConcurrentCache(size_t maxSize, TArgs... args) {
            for (auto&& shard : TBase::Shards) {
                shard.Container = MakeHolder<TCacheImpl>(maxSize / ShardsCount, std::forward<TArgs>(args)...);
            }
        }

        TMaybe<TValue> find(const TKey& key) {
            TShard& shard = GetShard(key);
            TSharedGuard guard(shard.Lock);
            auto p = shard.Container->Find(key);
            if (p == shard.Container->End()) {
                return {};
            }
            Y_ASSERT(p.Key() == key);
            return p.Value();
        }

        bool insert(const std::pair<TKey, TValue>& keyValue) {
            return emplace(keyValue.first, keyValue.second);
        }
        bool emplace(const TKey& key, const TValue& value) {
            TShard& shard = GetShard(key);
            TUniqueGuard guard(shard.Lock);
            return shard.Container->Insert(key, value);
        }
        bool erase(const TKey& key) {
            TShard& shard = GetShard(key);
            TUniqueGuard guard(shard.Lock);
            auto p = shard.Container->FindWithoutPromote(key);
            if (p == shard.Container->End()) {
                return false;
            }
            shard.Container->Erase(p);
            return true;
        }
        void update(const TKey& key, const TValue& value) {
            TShard& shard = GetShard(key);
            TUniqueGuard guard(shard.Lock);
            shard.Container->Update(key, value);
        }

    private:
        using TBase::GetShard;
    };
}
