#pragma once

#include <util/digest/multi.h>
#include <util/system/mutex.h>
#include <util/system/rwlock.h>
#include <util/system/spinlock.h>

#include <array>

namespace NUtil {
    struct TAdaptiveLocker {
        using TLock = TAdaptiveLock;
        using TSharedGuard = TGuard<TAdaptiveLock>;
        using TUniqueGuard = TGuard<TAdaptiveLock>;
    };

    struct TMutexLocker {
        using TLock = TMutex;
        using TSharedGuard = TGuard<TMutex>;
        using TUniqueGuard = TGuard<TMutex>;
    };

    struct TRWMutexLocker {
        using TLock = TRWMutex;
        using TSharedGuard = TReadGuard;
        using TUniqueGuard = TWriteGuard;
    };

    template <
        class Key,
        class TContainer,
        size_t ShardsCount = 64,
        class TLocker = TAdaptiveLocker
    >
    class TConcurrentSharded {
    protected:
        using TLock = typename TLocker::TLock;
        using TSharedGuard = typename TLocker::TSharedGuard;
        using TUniqueGuard = typename TLocker::TUniqueGuard;

    protected:
        inline TConcurrentSharded() = default;

    protected:
        static constexpr size_t ShardSize = 64;
        struct alignas(ShardSize) TShard: TNonCopyable {
            TContainer Container;
            TLock Lock;
        };
        static_assert(sizeof(TShard) >= ShardSize, "incorrect padding");
        static_assert(alignof(TShard) >= ShardSize, "incorrect align");

    protected:
        const TShard& GetShard(const Key& key) const {
            const size_t index = GetShardIndex(key);
            Y_ASSERT(index < Shards.size());
            return Shards[index];
        }
        TShard& GetShard(const Key& key) {
            const size_t index = GetShardIndex(key);
            Y_ASSERT(index < Shards.size());
            return Shards[index];
        }
        size_t GetShardIndex(const Key& key) const {
            return MultiHash(key, ShardsCount) % ShardsCount;
        }

    protected:
        std::array<TShard, ShardsCount> Shards;
    };
}
