#pragma once

#include "concurrent_common.h"

#include <util/generic/hash.h>

#include <array>

namespace NUtil {
    template <
        class Key, class T,
        size_t ShardsCount = 64,
        class TLocker = TAdaptiveLocker,
        class THash = THashMap<Key, T>
    >
    class TConcurrentHashMap: public TConcurrentSharded<Key, THash, ShardsCount, TLocker> {
    private:
        using TBase = TConcurrentSharded<Key, THash, ShardsCount, TLocker>;
        using TShard = typename TBase::TShard;
        using TSharedGuard = typename TBase::TSharedGuard;
        using TUniqueGuard = typename TBase::TUniqueGuard;

        using TConstIterator = typename THash::const_iterator;
        using TIterator = typename THash::iterator;
        using TValue = typename THash::value_type;

    public:
        template <class CIterator, class CValue, class CGuard>
        class TBaseHandleProxy: public TPointerBase<TBaseHandleProxy<CIterator, CValue, CGuard>, CValue> {
        public:
            TBaseHandleProxy(CIterator&& iterator, CIterator&& end, CGuard&& guard) noexcept
                : Iterator(iterator)
                , End(end)
                , Guard(std::forward<CGuard>(guard))
            {
            }

            CValue* Get() const noexcept {
                Y_VERIFY(Guard);
                if (Iterator != End) {
                    return &(*Iterator);
                } else {
                    return nullptr;
                }
            }

            CIterator Raw() {
                Y_VERIFY(Guard);
                return Iterator;
            }

        private:
            CIterator Iterator;
            CIterator End;
            CGuard Guard;
        };

        template <class CValue, class CGuard>
        class TBaseValueProxy: public TPointerBase<TBaseValueProxy<CValue, CGuard>, CValue> {
        public:
            TBaseValueProxy(CValue& value, CGuard&& guard) noexcept
                : Value(value)
                , Guard(std::forward<CGuard>(guard))
            {
            }

            CValue* Get() const noexcept {
                Y_VERIFY(Guard);
                return &Value;
            }

        private:
            CValue& Value;
            CGuard Guard;
        };

        using TConstHandleProxy = TBaseHandleProxy<TConstIterator, const TValue, TSharedGuard>;
        using THandleProxy = TBaseHandleProxy<TIterator, TValue, TUniqueGuard>;

        using TConstValueProxy = TBaseValueProxy<const T, TSharedGuard>;
        using TValueProxy = TBaseValueProxy<T, TUniqueGuard>;

    public:
        TConstValueProxy at(const Key& key) const {
            const TShard& shard = GetShard(key);
            TSharedGuard guard(shard.Lock);
            return { shard.Container.at(key), std::move(guard) };
        }
        TValueProxy at(const Key& key) {
            TShard& shard = GetShard(key);
            TUniqueGuard guard(shard.Lock);
            return { shard.Container.at(key), std::move(guard) };
        }
        TConstValueProxy operator[](const Key& key) const {
            const TShard& shard = GetShard(key);
            TSharedGuard guard(shard.Lock);
            return { shard.Container[key], std::move(guard) };
        }
        TValueProxy operator[](const Key& key) {
            TShard& shard = GetShard(key);
            TUniqueGuard guard(shard.Lock);
            return { shard.Container[key], std::move(guard) };
        }

        TConstHandleProxy find(const Key& key) const {
            const TShard& shard = GetShard(key);
            TSharedGuard guard(shard.Lock);
            return { shard.Container.find(key), shard.Container.end(), std::move(guard) };
        }
        THandleProxy find(const Key& key) {
            TShard& shard = GetShard(key);
            TUniqueGuard guard(shard.Lock);
            return { shard.Container.find(key), shard.Container.end(), std::move(guard) };
        }
        auto erase(THandleProxy&& i) {
            TShard& shard = GetShard(i->first);
            return shard.Container.erase(i.Raw());
        }
        auto erase(const Key& key) {
            TShard& shard = GetShard(key);
            TUniqueGuard guard(shard.Lock);
            return shard.Container.erase(key);
        }

        template <class... TArgs>
        std::pair<THandleProxy, bool> emplace(const Key& key, TArgs&&... args) {
            TShard& shard = GetShard(key);
            TUniqueGuard guard(shard.Lock);
            auto emplacement = shard.Container.emplace(key, std::forward<TArgs>(args)...);
            return {
                THandleProxy(std::move(emplacement.first), shard.Container.end(), std::move(guard)),
                emplacement.second
            };
        }
        std::pair<THandleProxy, bool> insert(const typename THash::value_type& value) {
            TShard& shard = GetShard(value.first);
            TUniqueGuard guard(shard.Lock);
            auto insertion = shard.Container.insert(value);
            return {
                THandleProxy(std::move(insertion.first), shard.Container.end(), std::move(guard)),
                insertion.second
            };
        }

    private:
        using TBase::GetShard;
    };
}
