#pragma once

#include <util/generic/maybe.h>
#include <util/generic/ptr.h>
#include <util/system/spinlock.h>

template <class T>
class TThreadSafeContainer {
public:
    using TValueType = T;

public:
    TThreadSafeContainer() noexcept = default;
    TThreadSafeContainer(T&& value) noexcept
        : Value(std::move(value))
    {
    }
    TThreadSafeContainer(const TThreadSafeContainer<T>& other) = delete;
    TThreadSafeContainer(TThreadSafeContainer<T>&& other) = default;

    explicit operator bool() const noexcept {
        return Value.Defined();
    }

    void Consume(T&& value) noexcept {
        auto guard = Guard(Lock);
        Value = std::move(value);
    }
    TMaybe<T> Release() noexcept {
        auto guard = Guard(Lock);
        auto result = std::move(Value);
        Value.Clear();
        return result;
    }

private:
    TAdaptiveLock Lock;
    TMaybe<T> Value;
};

template <class T>
using TThreadSafeContainerPtr = TAtomicSharedPtr<TThreadSafeContainer<T>>;

template <class T>
auto MakeThreadSafeContainer() noexcept {
    return MakeAtomicShared<TThreadSafeContainer<T>>();
}

template <class T>
auto MakeThreadSafeContainer(T&& value) noexcept {
    using TT = std::remove_reference_t<T>;
    return MakeAtomicShared<TThreadSafeContainer<TT>>(std::forward<T>(value));
}
