#include "eventlogger.h"

#include <rtline/library/json/builder.h>
#include <rtline/library/json/cast.h>

#include <util/system/tls.h>

Y_POD_STATIC_THREAD(NThreading::TEventLoggerPtr) ThreadEventLogger;

void NThreading::IEventLogger::AddEvent(NJson::TJsonValue&& ev) {
    AddEvent(Now(), std::move(ev));
}

NThreading::TEventLoggerGuard::TEventLoggerGuard(TEventLoggerPtr value)
    : PreviousValue(std::move(ThreadEventLogger))
{
    ThreadEventLogger = std::move(value);
}

NThreading::TEventLoggerGuard::~TEventLoggerGuard() {
    ThreadEventLogger = std::move(PreviousValue);
}

NThreading::TEventLoggerPtr NThreading::CaptureEventLogger() {
    return ThreadEventLogger;
}

NThreading::IEventLogger* NThreading::GetEventLogger() {
    auto result = TlsRef(ThreadEventLogger);
    return result && result->ShouldRecordEventLog() ? result.Get() : nullptr;
}

NThreading::TEventGuard::TEventGuard(const TString& name, IEventLogger& logger)
    : TEventGuardTraits(name)
    , Logger(logger)
{
    Logger.AddEvent(Start, BuildStartEvent());
}

NThreading::TEventGuardTraits::TEventGuardTraits(const TString& name)
    : Name(name)
    , Start(Now())
{
}

NJson::TJsonValue NThreading::TEventGuardTraits::BuildStartEvent() {
    return NJson::TMapBuilder
        ("event", "start")
        ("source", Name)
    ;
}

NJson::TJsonValue NThreading::TEventGuardTraits::BuildFinishEvent(TInstant finish) {
    NJson::TJsonValue ev = NJson::TMapBuilder
        ("event", "finish")
        ("source", Name)
        ("duration", NJson::ToJson(finish - Start))
    ;
    if (auto uncaughtExceptions = std::uncaught_exceptions()) {
        ev["uncaught_exceptions"] = uncaughtExceptions;
    }
    return ev;
}

NThreading::TEventGuard::~TEventGuard() {
    auto finish = Now();
    Logger.AddEvent(finish, BuildFinishEvent(finish));
}

NThreading::TOptionalEventGuard NThreading::BuildEventGuard(const TString& name) {
    auto eventLogger = GetEventLogger();
    if (!eventLogger) {
        return {};
    }
    return MakeMaybe<TEventGuard>(name, *eventLogger);
}
