#pragma once

#include "container.h"

#include <rtline/library/json/exception.h>
#include <rtline/util/types/exception.h>

#include <library/cpp/threading/future/future.h>

#include <util/system/type_name.h>

namespace NThreading {
    template <class E = yexception>
    class TExceptionFuture {
    public:
        template <class... TArgs>
        TExceptionFuture(TArgs&&... args)
            : Value(std::forward<TArgs>(args)...)
        {
        }

        template <class T>
        TExceptionFuture& operator<<(const T& t) {
            Value << t;
            return *this;
        }

        template <class T>
        operator TFuture<T>() {
            return MakeErrorFuture<T>(std::make_exception_ptr(std::move(Value)));
        }

    private:
        E Value;
    };

    class TUninitialized {
    public:
        template <class T>
        inline operator TFuture<T>() const {
            return {};
        }
    };
    inline constexpr TUninitialized Uninitialized = {};

    template <class T>
    using TFutures = TVector<TFuture<T>>;

    template <class T>
    TFuture<T> Initialize(const TFuture<T>& value) {
        if (value.Initialized()) {
            return value;
        } else {
            return TExceptionFuture<TWithBackTrace<yexception>>() << TypeName(value) << " is not initialized";
        }
    }

    template <class T>
    TFuture<TVector<T>> Merge(TFutures<T>&& values, bool ignoreExceptions = false) {
        auto waiter = ignoreExceptions ? NThreading::WaitAll(values) : NThreading::WaitExceptionOrAll(values);
        return waiter.Apply([ignoreExceptions, values = std::move(values)](const TFuture<void>& /*w*/) mutable {
            TVector<T> result;
            result.reserve(values.size());
            for (auto&& value : values) {
                if (value.HasException() && ignoreExceptions) {
                    continue;
                }
                result.push_back(value.ExtractValue());
            }
            return result;
        });
    }

    template <class T>
    std::exception_ptr GetException(const TFuture<T>& future) {
        try {
            future.GetValue();
            return nullptr;
        } catch (...) {
            return std::current_exception();
        }
    }

    template <class T>
    NJson::TJsonValue GetExceptionInfo(const TFuture<T>& future) {
        try {
            future.GetValue();
            return {};
        } catch (const TCodedException& e) {
            return e.GetReport();
        } catch (const yexception& e) {
            return NJson::GetExceptionInfo(e);
        } catch (const std::exception& e) {
            return NJson::GetExceptionInfo(e);
        } catch (...) {
            return "unknown exception";
        }
    }

    template <class T>
    TString GetExceptionMessage(const TFuture<T>& future) {
        try {
            future.GetValue();
            return {};
        } catch (const TCodedException& e) {
            return e.GetReport().GetStringRobust();
        } catch (const std::exception& e) {
            return e.what();
        } catch (...) {
            return "unknown exception";
        }
    }

    namespace NPrivate {
        void Schedule(TInstant timestamp, std::function<void()>&& f);
    }

    template <class T, class F>
    void Subscribe(const TFuture<T>& future, F&& callback, TInstant deadline) {
        auto now = Now();
        if (now >= deadline) {
            callback(future);
            return;
        }
        auto callbackContainer = MakeThreadSafeContainer(std::forward<F>(callback));
        NPrivate::Schedule(deadline, [callbackContainer, future] {
            auto optionalCallback = callbackContainer->Release();
            if (optionalCallback) {
                const auto& callback = *optionalCallback;
                callback(future);
            }
        });
        future.Subscribe([callbackContainer = std::move(callbackContainer)](const TFuture<T>& future) {
            auto optionalCallback = callbackContainer->Release();
            if (optionalCallback) {
                const auto& callback = *optionalCallback;
                callback(future);
            }
        });
    }

    class TLazyPromise {
    public:
        TFuture<void> GetFuture();

        bool SetValue();

    private:
        TPromise<void> Promise;
    };

    class TGuard {
    public:
        ~TGuard() {
            Promise.SetValue();
        }

        TFuture<void> GetFuture() {
            return Promise.GetFuture();
        }

    private:
        TLazyPromise Promise;
    };
}
