#include "requester.h"

#include <rtline/library/json/parse.h>
#include <rtline/util/types/exception.h>

#include <library/cpp/http/misc/httpcodes.h>
#include <library/cpp/json/json_reader.h>
#include <library/cpp/logger/global/global.h>
#include <library/cpp/neh/http_common.h>
#include <library/cpp/string_utils/url/url.h>

#include <util/charset/utf8.h>
#include <util/random/random.h>
#include <util/stream/file.h>
#include <util/string/builder.h>
#include <util/string/join.h>
#include <util/string/strip.h>

#include <tuple>

NNeh::TResponseRef NDrive::TBaseRequester::Req(const TString& request) const {
    TString r = request;
    if (GetHostAndPort(request).empty()) {
        TStringBuilder url;
        url << "http://" << HostPort;
        if (!r.StartsWith('/')) {
            url << '/';
        }
        url << r;
        r = std::move(url);
    }
    if (ExtraCgi) {
        if (r.find('?') == TString::npos) {
            r.append('?');
        } else {
            r.append('&');
        }
        r.append(ExtraCgi);
    }
    NNeh::TMessage message = NNeh::TMessage::FromString(r);
    NNeh::NHttp::MakeFullRequest(message, Headers, {});
    for (ui32 attempt = 0; attempt < 1; ++attempt) {
        NNeh::THandleRef handle = NNeh::Request(message);
        if (handle) {
            DEBUG_LOG << "Requesting " << request << Endl;
            const NNeh::TResponseRef response = handle->Wait(Options.Timeout);
            if (response) {
                const ui32 code = response->IsError() ? response->GetErrorCode() : HTTP_OK;
                if (IsServerError(code)) {
                    WARNING_LOG << "code " << code << ": " << request << Endl;
                } else {
                    return response;
                }
            } else {
                WARNING_LOG << "empty response: " << request << Endl;
            }
        }
        CHECK_WITH_LOG(!Options.BackoffTable.empty());
        TDuration backoffCap = attempt < Options.BackoffTable.size() ? Options.BackoffTable[attempt] : Options.BackoffTable.back();
        TDuration backoff = TDuration::MicroSeconds(RandomNumber<double>() * backoffCap.MicroSeconds());
        Sleep(backoff);
    }
    ythrow yexception() << "cannot access " << request;
}

NJson::TJsonValue NDrive::TBaseRequester::RequestJson(const TString& request) const {
    const auto response = Req(request);
    Y_ENSURE(response);
    const ui32 code = response->IsError() ? response->GetErrorCode() : HTTP_OK;
    if (code == HTTP_OK) {
        NJson::TJsonValue json;
        NJson::ReadJsonFastTree(response->Data, &json, true);
        return json;
    } else {
        ythrow TCodedException(code) << request << "\t" << code << '\t' << response->GetSystemErrorCode() << '\t' << response->GetErrorText() << '\t' << response->Data;
    }
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::TSessionRequester::TOffer& result) {
    result.Raw = value;
    return NJson::ParseField(value["OfferId"], result.Id);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::TSessionRequester::TTag& result) {
    return NJson::ParseField(value["tag"], result.Name, true) &&
        NJson::ParseField(value["tag_id"], result.Id) &&
        NJson::ParseField(value["performer"], result.Performer) &&
        NJson::ParseField(value["priority"], result.Priority);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::TSessionRequester::TTagEvent& result) {
    return
        NJson::ParseField(value["action"], result.HistoryAction) &&
        NJson::ParseField(value["event_id"], result.HistoryEventId) &&
        NJson::ParseField(value["tag_name"], result.TagName) &&
        NJson::ParseField(value["timestamp"], result.HistoryTimestamp) &&
        NJson::ParseField(value["user_id"], result.HistoryUserId);
}

namespace {
    NDrive::TSessionRequester::TCar ParseCar(const NJson::TJsonValue& car, const TSet<TString>& trackedTags, bool trackSpecialTags) {
        const TString& carId = car["id"].GetStringRobust();
        const ui64 imei = car["imei"].GetUIntegerRobust();
        const TString& model = car["model_id"].GetStringRobust();
        const TString& status = car["status"].GetString();
        const auto optionalTags = NJson::FromJson<TMaybe<NDrive::TSessionRequester::TTags>>(car["tags"]);

        NDrive::TSessionRequester::TTags tags;
        if (optionalTags) {
            for (auto&& tag : *optionalTags) {
                if (trackedTags.contains(tag.Name)) {
                    tags.push_back(std::move(tag));
                    continue;
                }
                if (trackSpecialTags && tag.Performer && tag.Priority > 0) {
                    tags.push_back(std::move(tag));
                    continue;
                }
            }
        }

        ui64 index = 0;
        return {
            carId,
            index,
            imei,
            model,
            status,
            std::move(tags)
        };
    }

    void ProcessEvent(
        TInstant start,
        const TString& action,
        const TString& carId,
        const TString& sessionId,
        const TString& tagName,
        const TString& userId,
        TVector<NDrive::TStatus>& result
    ) {
        Y_ENSURE(action);
        if (action == "evolve" && userId == "robot-frontend") {
            return;
        }
        if (action == "update_data") {
            return;
        }
        if (action != "set_performer" && !result.empty() && result.back().GetSessionId() == sessionId) {
            result.back().SetFinish(start);
        }

        constexpr TStringBuf oldStatePrefix = "old_state_";
        const TString& carStatus = tagName.StartsWith(oldStatePrefix) ? tagName.substr(oldStatePrefix.size()) : tagName;
        if (action != "drop_performer") {
            const TString& actionId = ToString(start.Seconds()) + "-" + action + "-" + sessionId;
            NDrive::TStatus status(carStatus, actionId, carId, sessionId, userId, start);
            result.push_back(std::move(status));
        }
    }

    TString GetDriveToken() try {
        TFsPath path = "~/.drive/staff";
        Y_ENSURE(path.Exists(), path << " does not exist");
        return Strip(TIFStream(path).ReadAll());
    } catch (const std::exception& e) {
        throw yexception() << "cannot get drive token from environment: " << FormatExc(e);
    }
}

NDrive::TSessionRequester::TSessionRequester(const TString& hostPort, const TString& extraCgi, const TString& oauthToken, ui64 quota)
    : TBaseRequester(hostPort, extraCgi, "Authorization: OAuth " + (oauthToken ? oauthToken : GetDriveToken()) + "\r\nUserPermissionsCache: true")
    , Quoter(quota ? MakeHolder<TBucketQuoter<int>>(quota, 3 * quota) : nullptr)
{
}

TMap<TString, NDrive::IRequester::TCar> NDrive::TSessionRequester::GetCars(TConstArrayRef<TString> ids) const {
    auto request = TStringBuilder() << "http://" << HostPort << "/api/staff/car/list?traits=ReportIMEI";
    if (TrackSpecialTags || !TrackedTags.empty()) {
        request << ",ReportTagDetails";
    }
    if (ids) {
        request << "&car_id=" << JoinStrings(ids.begin(), ids.end(), ",");
    }
    if (TagFilter) {
        request << "&tags_filter=" << TagFilter;
    }
    if (Options.Timeout) {
        request << "&timeout=" << Options.Timeout.MicroSeconds();
    }
    const NJson::TJsonValue response = RequestJson(request);

    TMap<TString, NDrive::TSessionRequester::TCar> result;
    for (auto&& car : response["cars"].GetArraySafe()) {
        auto object = ParseCar(car, TrackedTags, TrackSpecialTags);
        result.emplace(object.Id, object);
    }
    return result;
}

TMap<ui64, NDrive::IRequester::TCar> NDrive::TSessionRequester::GetCars(TConstArrayRef<ui64> imeis) const {
    auto request = TStringBuilder() << "http://" << HostPort << "/api/staff/car/list?traits=ReportIMEI";
    if (TrackSpecialTags || !TrackedTags.empty()) {
        request << ",ReportTagDetails";
    }
    if (imeis) {
        request << "&imei=" << JoinStrings(imeis.begin(), imeis.end(), ",");
    }
    if (Options.Timeout) {
        request << "&timeout=" << Options.Timeout.MicroSeconds();
    }
    NJson::TJsonValue response;
    try {
        response = RequestJson(request);
    } catch (const TCodedException& e) {
        if (e.GetCode() == HTTP_NOT_FOUND) {
            return {};
        }
        throw;
    }

    TMap<ui64, NDrive::TSessionRequester::TCar> result;
    for (auto&& car : response["cars"].GetArraySafe()) {
        auto object = ParseCar(car, TrackedTags, TrackSpecialTags);
        result.emplace(object.IMEI, object);
    }
    return result;
}

TVector<NDrive::TStatus> NDrive::TSessionRequester::GetDelta(TInstant from, TInstant to, TConstArrayRef<TCar> cars, bool reportActiveSessions) const {
    if (Quoter) {
        Quoter->Sleep();
        Quoter->Use(1);
    }
    auto request = TStringBuilder() << "http://" << HostPort << "/api/staff/sessions?since=" << from.Seconds() << "&until=" << to.Seconds();
    if (!cars.empty()) {
        request << "&imei=";
        for (size_t i = 0; i < cars.size(); ++i) {
            if (i) {
                request << ',';
            }
            request << cars[i].IMEI;
        }
    }
    if (reportActiveSessions) {
        request << "&report_active_sessions=1";
    }
    if (Options.Timeout) {
        request << "&timeout=" << Options.Timeout.MicroSeconds();
    }
    NJson::TJsonValue response = RequestJson(request);

    TVector<NDrive::TStatus> result;
    for (auto&& session : response["sessions"].GetArraySafe()) {
        Y_ENSURE(session["meta"].IsMap());
        Y_ENSURE(session["timeline"].IsArray());
        const TString& carId = session["meta"]["object_id"].GetStringSafe();
        const TString& sessionId = session["meta"]["session_id"].IsString()
            ? session["meta"]["session_id"].GetStringSafe()
            : "unknown";
        for (auto&& element : session["timeline"].GetArraySafe()) {
            const TInstant start = TInstant::Seconds(element["timestamp"].GetUIntegerRobust());
            const TString& action = element["action"].GetStringSafe();
            const TString& userId = element["user_id"].GetStringSafe();
            const TString& tagName = element["tag_name"].GetStringSafe();
            if (action != "set_performer" && !result.empty() && result.back().GetSessionId() == sessionId) {
                if (result.back().GetStart() > start) {
                    ERROR_LOG << "bad sorting in "
                        << "from=" << from.MicroSeconds() << ' '
                        << "to=" << to.MicroSeconds() << ' '
                        << "cars_count=" << cars.size() << ' '
                        << "reportActiveSessions=" << reportActiveSessions << ' '
                        << response.GetStringRobust() << Endl;
                }
            }
            ProcessEvent(start, action, carId, sessionId, tagName, userId, result);
        }
    }
    return result;
}

TMap<TString, TString> NDrive::TSessionRequester::GetHeadId(TConstArrayRef<TString> carIds) const {
    TString request = "http://" + HostPort + "/api/staff/head/info";
    if (carIds) {
        request += "?car_id=" + JoinStrings(carIds.begin(), carIds.end(), ",");
    }
    const NJson::TJsonValue response = RequestJson(request);

    TMap<TString, TString> result;
    for (auto&& object : response["objects"].GetArraySafe()) {
        const TString& carId = object["car_id"].GetStringSafe();
        const TString& headId = object["head_id"].GetStringSafe();
        result.emplace(carId, headId);
    }
    return result;
}

NDrive::TSessionRequester::TSession NDrive::TSessionRequester::GetSession(const TString& sessionId) const {
    auto request = TStringBuilder() << "http://" << HostPort << "/api/staff/sessions/history?session_id=" << sessionId;
    auto response = RequestJson(request);

    const auto& sessions = response["sessions"];
    Y_ENSURE(sessions.IsArray(), "sessions is not an array: " << sessions.GetStringRobust());
    Y_ENSURE(sessions.GetArray().size() > 0, "sessions is of incorrect size: " << sessions.GetStringRobust());

    auto findSession = [&] (const TString& id) {
        for (auto&& session : sessions.GetArray()) {
            if (session["segment"]["meta"]["session_id"].GetString() == id) {
                return session;
            }
            if (session["segment"]["session_id"].GetString() == id) {
                return session;
            }
        }
        ythrow yexception() << "cannot find session " << id;
    };
    const auto& session = findSession(sessionId);

    auto car = ParseCar(session["car"], TrackedTags, TrackSpecialTags);
    Y_ENSURE(car.Id);
    Y_ENSURE(car.IMEI);

    const auto& carId = car.Id;
    const auto& userId = session["user_details"]["id"].GetString();
    Y_ENSURE(carId);
    Y_ENSURE(userId);

    auto offer = NJson::FromJson<TMaybe<TOffer>>(session["offer_proto"]);

    const auto& events = session["segment"]["events"];
    Y_ENSURE(events.IsArray(), "events is not an array: " << events.GetStringRobust());
    TVector<TStatus> statuses;
    TVector<TTagEvent> tagEvents;
    for (auto&& element : events.GetArray()) {
        const TInstant start = TInstant::Seconds(element["timestamp"].GetUIntegerRobust());
        const TString& action = element["action"].GetString();
        const TString& tagName = element["tag_name"].GetString();
        auto tagEvent = NJson::FromJson<NDrive::TSessionRequester::TTagEvent>(element);
        tagEvents.push_back(std::move(tagEvent));
        ProcessEvent(start, action, carId, sessionId, tagName, userId, statuses);
    }
    std::sort(tagEvents.begin(), tagEvents.end());

    return {
        std::move(car),
        std::move(offer),
        std::move(statuses),
        std::move(tagEvents)
    };
}

NJson::TJsonValue NDrive::TSessionRequester::Request(const TString& query) const {
    return RequestJson("http://" + HostPort + query);
}

bool NDrive::TSessionRequester::TCar::operator!=(const TCar& other) const {
    auto tagsSize = Tags.size();
    auto otherTagsSize = other.Tags.size();
    if (std::tie(Id, IMEI, Model, tagsSize) != std::tie(other.Id, other.IMEI, other.Model, otherTagsSize)) {
        return true;
    }
    auto i = Tags.begin();
    auto j = other.Tags.begin();
    for (; i != Tags.end() && j != other.Tags.end(); ++i, ++j) {
        if (*i != *j) {
            return true;
        }
    }
    return false;
}

TSet<TString> NDrive::TSessionRequester::TCar::GetServicePerformers(const TSet<TString>* excludedTags) const {
    TSet<TString> result;
    for (auto&& tag : Tags) {
        if (excludedTags && excludedTags->contains(tag.Name)) {
            continue;
        }
        if (tag.Performer && tag.Priority > 0) {
            result.insert(tag.Performer);
        }
    }
    return result;
}

const NDrive::TSessionRequester::TTag* NDrive::TSessionRequester::TCar::GetTag(TStringBuf name) const {
    for (auto&& tag : Tags) {
        if (tag.Name == name) {
            return &tag;
        }
    }
    return nullptr;
}
