#pragma once

#include "status.h"

#include <rtline/util/types/timestamp.h>

#include <library/cpp/bucket_quoter/bucket_quoter.h>
#include <library/cpp/json/json_value.h>
#include <library/cpp/neh/neh.h>

#include <util/generic/array_ref.h>
#include <util/generic/set.h>
#include <util/system/rwlock.h>

namespace NDrive {
    class TBaseRequester {
    public:
        struct TOptions {
            TVector<TDuration> BackoffTable = {
                TDuration::MilliSeconds(500),
                TDuration::Seconds(1),
                TDuration::Seconds(5),
                TDuration::Seconds(10)
            };
            TDuration Timeout = TDuration::Seconds(42);
        };

    public:
        TBaseRequester(const TString& hostPort, const TString& extraCgi = {}, const TString headers = {})
            : HostPort(hostPort)
            , ExtraCgi(extraCgi)
            , Headers(headers)
        {
        }

        TString GetDescription() const {
            return HostPort + "/?" + ExtraCgi;
        }

    protected:
        NNeh::TResponseRef Req(const TString& request) const;
        NJson::TJsonValue RequestJson(const TString& request) const;

    protected:
        const TString HostPort;
        const TString ExtraCgi;
        const TString Headers;
        const TOptions Options;
    };

    class IRequester {
    public:
        struct TTag {
            TString Id;
            TString Name;
            TString Performer;
            i64 Priority;

            bool operator!=(const TTag& other) const {
                return Id != other.Id;
            }

            Y_SAVELOAD_DEFINE(
                Id,
                Name,
                Performer,
                Priority
            );
        };
        using TTags = TVector<TTag>;

        struct TTagEvent {
            ui64 HistoryEventId = 0;
            TString HistoryAction;
            TTimestamp HistoryTimestamp;
            TString HistoryUserId;
            TString TagName;

            inline bool operator< (const TTagEvent& other) const {
                return HistoryEventId < other.HistoryEventId;
            }
        };

        struct TCar {
            TString Id;
            ui64 Index = 0;
            ui64 IMEI;
            TString Model;
            TString Status;
            TTags Tags;

            bool operator!=(const TCar& other) const;

            TSet<TString> GetServicePerformers(const TSet<TString>* excludedTags = nullptr) const;
            const TTag* GetTag(TStringBuf name) const;

            Y_SAVELOAD_DEFINE(
                Id,
                Index,
                IMEI,
                Model,
                Status,
                Tags
            );
        };

        struct TOffer {
            TString Id;

            NJson::TJsonValue Raw;
        };

        struct TSession {
            TCar Car;
            TMaybe<TOffer> Offer;
            TVector<TStatus> Statuses;
            TVector<TTagEvent> TagEvents;
        };

    public:
        virtual ~IRequester() = default;

        virtual TMap<TString, TCar> GetCars(TConstArrayRef<TString> ids = {}) const = 0;
        virtual TMap<ui64, TCar> GetCars(TConstArrayRef<ui64> imeis) const = 0;
        virtual TVector<TStatus> GetDelta(TInstant from, TInstant to, TConstArrayRef<TCar> cars = {}, bool reportActiveSessions = false) const = 0;
    };
    using TRequesterPtr = TAtomicSharedPtr<IRequester>;

    class TSessionRequester
        : public IRequester
        , public TBaseRequester
    {
    public:
        TSessionRequester(const TString& hostPort, const TString& extraCgi, const TString& oauthToken = {}, ui64 quota = 0);

        void SetTagFilter(const TString& value) {
            TagFilter = value;
        }
        void SetTrackedTags(TSet<TString> tags) {
            TrackedTags = std::move(tags);
        }
        void SetTrackSpecialTags(bool value) {
            TrackSpecialTags = value;
        }

        TMap<TString, TCar> GetCars(TConstArrayRef<TString> ids = {}) const override;
        TMap<ui64, TCar> GetCars(TConstArrayRef<ui64> imeis) const override;
        TVector<TStatus> GetDelta(TInstant from, TInstant to, TConstArrayRef<TCar> cars = {}, bool reportActiveSessions = false) const override;

        TMap<TString, TString> GetHeadId(TConstArrayRef<TString> carIds) const;
        TSession GetSession(const TString& sessionId) const;

        NJson::TJsonValue Request(const TString& query) const;

    private:
        THolder<TBucketQuoter<int>> Quoter;
        TString TagFilter;
        TSet<TString> TrackedTags;
        bool TrackSpecialTags = false;
    };
}
