#include "status.h"

NDrive::TStatus::TStatus(const NJson::TJsonValue& json) {
    if (json.Has("car_status")) {
        CarStatus = json["car_status"].GetStringSafe();
    } else if (json.Has("status")) {
        CarStatus = json["status"].GetStringSafe();
    } else {
        ythrow yexception() << "cannot find car status in " << json.GetStringRobust();
    }

    if (json.Has("car_id")) {
        CarId = json["car_id"].GetStringSafe();
    } else if (json.Has("id")) {
        CarId = json["id"].GetStringSafe();
    } else {
        ythrow yexception() << "cannot find car id in " << json.GetStringRobust();
    }

    if (json.Has("timestamp_start") && !json["timestamp_start"].IsNull()) {
        auto start = json["timestamp_start"].GetDoubleRobust();
        Start = TInstant::MicroSeconds(1000000 * start);
    } else {
        Start = TInstant::Zero();
    }
    if (json.Has("timestamp") && !json["timestamp"].IsNull()) {
        auto start = json["timestamp"].GetDoubleRobust();
        Start = TInstant::MicroSeconds(1000000 * start);
    }
    if (json.Has("timestamp_finish") && !json["timestamp_finish"].IsNull()) {
        auto finish = json["timestamp_finish"].GetDoubleRobust();
        Finish = TInstant::MicroSeconds(1000000 * finish);
    }

    if (json.Has("imei") && !json["imei"].IsNull()) {
        IMEI = json["imei"].GetUIntegerRobust();
    }

    ActionId = json["action_id"].GetString();
    SessionId = json["session_id"].GetString();
    UserId = json["user_id"].GetString();

    if (CarStatus != "ride") {
        ActionPrefix = CarStatus;
    }
}

TInterval<ui64> NDrive::TStatus::GetInterval() const {
    auto start = Start.MicroSeconds();
    auto finish = Finish.GetOrElse(TInstant::Max()).MicroSeconds();
    return { start, finish };
}

bool NDrive::TStatus::operator==(const TStatus& other) const {
    return
        CarStatus == other.CarStatus &&
        Start == other.Start &&
        Finish == other.Finish &&
        ActionId == other.ActionId
    ;
}

bool NDrive::TStatus::operator<(const TStatus& other) const {
    return std::make_pair(Start, Finish.GetOrElse(TInstant::Max())) < std::make_pair(other.Start, other.Finish.GetOrElse(TInstant::Max()));
}

bool NDrive::TStatus::operator<(TInstant timestamp) const {
    return Start < timestamp;
}

NDrive::TStatus NDrive::TStatus::Incorrect(ui64 imei, const TString& carId, ui64 index) {
    NDrive::TStatus result;
    result.SetCarStatus("incorrect");
    result.CarId = carId;
    result.Index = index;
    if (imei) {
        result.IMEI = imei;
    }
    return result;
}

NDrive::TStatus NDrive::TStatus::Zero(ui64 imei, const TString& carId, ui64 index) {
    NDrive::TStatus result;
    result.SetCarStatus("available");
    result.Finish = TInstant::Zero();
    result.ActionId = "null";
    result.SessionId = "null";
    result.CarId = carId;
    result.UserId = "null";
    result.Index = index;
    if (imei) {
        result.IMEI = imei;
    }
    return result;
}

template <>
void Out<NDrive::TStatus>(IOutputStream& out, typename TTypeTraits<NDrive::TStatus>::TFuncParam value) {
    out << "imei=" << value.GetIMEI() << '\t';
    out << "car_status=" << value.GetCarStatus() << '\t';
    out << "start=" << value.GetStart().MicroSeconds() << '\t';
    if (value.IsCurrent()) {
        out << "finish=null" << '\t';
    } else {
        out << "finish=" << value.GetFinish().MicroSeconds() << '\t';
    }
    out << "action_id=" << value.GetActionId() << '\t';
    out << "car_id=" << value.GetCarId() << '\t';
    out << "session_id=" << value.GetSessionId() << '\t';
    out << "user_id=" << value.GetUserId() << '\t';
}
